// 性能优化：添加节点处理限制和缓存控制
const MAX_NODES = 10000; // 增加最大处理节点数以确保完整高亮
const MAX_CACHE_SIZE = 100; // 增加缓存大小
const BATCH_SIZE = 50; // 增加批处理大小以提高效率
const PROCESSING_DELAY = 50; // 减少处理延迟
const DEBOUNCE_DELAY = 300; // 减少防抖延迟
let highlightEnabled = true;
let currentHighlights = [];
let currentSettings = {
    autoHighlight: true,
    caseSensitive: false,
    globalMatch: true,
    performanceMode: false, // 默认关闭性能模式以确保完整高亮
    onlyCurrentTab: false // 新增：只在当前标签页应用
};
let processingTimeout = null;
let cachedRegexps = new Map();
let isProcessing = false;
let pendingUpdate = false;
let autoHighlightTimeout = null;
let processingQueue = []; // 添加处理队列
let cacheHits = 0;
let cacheMisses = 0;

// 添加百度文库支持
const WENKU_CONFIG = {
    contentSelectors: [
        '.reader-word-layer',     // 文字层
        '.reader-txt-layer',      // 文本层
        '.sf-edu-wenku-content',  // 教育版内容
        '.reader-page',           // 页面容器
        '.doc-reader',            // 文档阅读器
        '.ppt-page-item',         // PPT页面
        '.xs-page'                // 新样式页面
    ],
    observerConfig: {
        childList: true,
        subtree: true,
        characterData: true
    },
    iframeSelectors: [
        '#reader-iframe',         // 主阅读器iframe
        '.reader-iframe',         // 其他阅读器iframe
        '#previewContent'         // 预览iframe
    ]
};

// 添加扩展上下文检查函数
function isExtensionContextValid() {
    try {
        // 尝试访问chrome.runtime.id - 如果扩展上下文无效，这会抛出错误
        return !!chrome.runtime.id;
    } catch (e) {
        console.debug('扩展上下文已失效', e);
        return false;
    }
}

// 优化：缓存正则表达式
function getRegExp(text) {
    const key = `${text}-${currentSettings.caseSensitive}-${currentSettings.globalMatch}`;
    if (!cachedRegexps.has(key)) {
        cacheMisses++;
        if (cachedRegexps.size >= MAX_CACHE_SIZE) {
            // 清理最旧的一半缓存
            const keysToDelete = Array.from(cachedRegexps.keys()).slice(0, Math.floor(MAX_CACHE_SIZE / 2));
            keysToDelete.forEach(k => cachedRegexps.delete(k));
        }
        
        // 构建正则表达式标志
        let flags = currentSettings.globalMatch ? 'g' : '';
        if (!currentSettings.caseSensitive) {
            flags += 'i';
        }
        
        const escapedText = escapeRegExp(text);
        // 使用单词边界来提高匹配准确性
        const pattern = `(${escapedText})`;
        cachedRegexps.set(key, new RegExp(pattern, flags));
    } else {
        cacheHits++;
    }
    const regex = cachedRegexps.get(key);
    regex.lastIndex = 0;
    return regex;
}

// 转义正则表达式特殊字符
function escapeRegExp(string) {
    return string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
}

// 优化：使用DocumentFragment提高性能，避免闪烁
function processTextNode(textNode, highlights) {
    if (!textNode || !textNode.nodeValue) return false;
    
    // 安全检查：检查节点是否仍然在DOM中
    if (!textNode.parentNode || !document.contains(textNode.parentNode)) {
        return false;
    }
    
    // 安全检查：检查父元素是否可以安全地修改
    const parent = textNode.parentNode;
    const parentTagName = parent.tagName.toLowerCase();
    
    // 检查父元素是否是交互式元素或编辑区域（额外安全检查）
    if (parent.contentEditable === 'true' || 
        parent.isContentEditable || 
        ['input', 'textarea', 'select', 'option', 'iframe'].includes(parentTagName)) {
        return false;
    }
    
    // 检查是否在表单或交互式元素内
    let currentNode = parent;
    while (currentNode && currentNode !== document.body) {
        if (currentNode.tagName) {
            const tagName = currentNode.tagName.toLowerCase();
            if (['form', 'input', 'select', 'textarea', 'button', 'iframe'].includes(tagName)) {
                return false;
            }
            
            // 检查特定属性
            if (currentNode.contentEditable === 'true' || 
                currentNode.isContentEditable ||
                currentNode.getAttribute('role') === 'textbox' ||
                currentNode.getAttribute('role') === 'grid' ||
                currentNode.getAttribute('role') === 'gridcell') {
                return false;
            }
        }
        currentNode = currentNode.parentNode;
    }
    
    const text = textNode.nodeValue;
    let hasChanges = false;
    
    // 收集所有匹配项
    let matches = [];
    for (const highlight of highlights) {
        const regex = getRegExp(highlight.text);
        
        let match;
        let searchText = text;
        let offset = 0;
        
        // 重置正则表达式的lastIndex
        regex.lastIndex = 0;
        
        while ((match = regex.exec(searchText)) !== null) {
            matches.push({
                start: match.index + offset,
                end: match.index + offset + match[0].length,
                text: match[0],
                highlight: highlight
            });
            
            // 防止无限循环
            if (match[0].length === 0) {
                regex.lastIndex++;
            }
            
            // 如果不是全局匹配，只匹配第一个
            if (!currentSettings.globalMatch) {
                break;
            }
        }
    }
    
    // 如果没有匹配项，直接返回
    if (matches.length === 0) return false;
    
    // 尝试在安全模式下修改DOM
    try {
        // 按照起始位置排序匹配项
        matches.sort((a, b) => a.start - b.start);
        
        // 合并重叠的匹配项（优先保留较长的匹配）
        matches = matches.reduce((acc, curr) => {
            if (acc.length === 0) {
                acc.push(curr);
                return acc;
            }
            
            const last = acc[acc.length - 1];
            if (curr.start < last.end) {
                // 如果当前匹配比上一个匹配长，则替换
                if (curr.end - curr.start > last.end - last.start) {
                    acc[acc.length - 1] = curr;
                }
            } else {
                acc.push(curr);
            }
            return acc;
        }, []);
        
        // 创建文档片段
        const fragment = document.createDocumentFragment();
        let lastIndex = 0;
        
        // 处理每个匹配项
        for (const match of matches) {
            if (match.start > lastIndex) {
                fragment.appendChild(
                    document.createTextNode(text.slice(lastIndex, match.start))
                );
            }
            
            const mark = document.createElement('mark');
            mark.textContent = match.text;
            mark.style.cssText = `
                background-color: ${match.highlight.color};
                color: ${getContrastColor(match.highlight.color)};
                border-radius: 2px;
                padding: 0 2px;
                margin: 0 1px;
                transition: background-color 0.2s ease;
                box-shadow: 0 1px 2px rgba(0,0,0,0.1);
                text-decoration: none;
                display: inline;
                position: relative;
                pointer-events: none;
                z-index: 1;
            `;
            mark.dataset.highlightId = match.highlight.id;
            fragment.appendChild(mark);
            
            lastIndex = match.end;
            hasChanges = true;
        }
        
        // 添加剩余的文本
        if (lastIndex < text.length) {
            fragment.appendChild(
                document.createTextNode(text.slice(lastIndex))
            );
        }
        
        // 替换原始节点
        if (hasChanges && textNode.parentNode) {
            // 最终安全检查：确保父节点仍然在DOM中
            if (document.contains(textNode.parentNode)) {
                textNode.parentNode.replaceChild(fragment, textNode);
                return true;
            }
        }
    } catch (error) {
        console.debug('高亮过程中出现错误:', error);
        return false;
    }
    
    return false;
}

// 优化：计算文本颜色对比度
function getContrastColor(bgColor) {
    const r = parseInt(bgColor.slice(1, 3), 16);
    const g = parseInt(bgColor.slice(3, 5), 16);
    const b = parseInt(bgColor.slice(5, 7), 16);
    const brightness = (r * 299 + g * 587 + b * 114) / 1000;
    return brightness > 128 ? '#000000' : '#ffffff';
}

// 优化：使用更高效的选择器，避免重复处理
function removeHighlights() {
    const marks = document.getElementsByTagName('mark');
    const marksArray = Array.from(marks);
    
    for (const mark of marksArray) {
        if (mark && mark.parentNode) {
            const text = mark.textContent;
            const textNode = document.createTextNode(text);
            mark.parentNode.replaceChild(textNode, mark);
        }
    }
    
    document.body.normalize();
}

// 优化：分批处理节点，避免闪烁
function processNodesInBatches(nodes, highlights, batchSize = BATCH_SIZE) {
    return new Promise((resolve, reject) => {
        if (isProcessing) {
            processingQueue.push({ nodes, highlights, batchSize, resolve, reject });
            return;
        }
        
        // 节点数量检查（不强制限制，只记录）
        if (nodes.length > MAX_NODES) {
            console.debug(`节点数量(${nodes.length})较多，将分批处理`);
            // 不再强制截断节点，而是分批处理所有节点
        }
        
        isProcessing = true;
        let index = 0;
        let processedCount = 0;
        const startTime = performance.now();
        
        function processBatch() {
            try {
                const endIndex = Math.min(index + batchSize, nodes.length);
                let hasChanges = false;
                
                for (let i = index; i < endIndex; i++) {
                    // 移除时间限制，确保批次内的所有节点都被处理
                    const node = nodes[i];
                    if (!node || !shouldProcessNode(node)) continue;
                    
                    if (processTextNode(node, highlights)) {
                        hasChanges = true;
                        processedCount++;
                    }
                }
                
                index = endIndex;
                
                if (index < nodes.length) {
                    // 继续处理剩余节点，不受时间限制
                    setTimeout(processBatch, PROCESSING_DELAY);
                } else {
                    isProcessing = false;
                    resolve(processedCount);
                    
                    // 处理队列中的下一个任务
                    if (processingQueue.length > 0) {
                        const next = processingQueue.shift();
                        processNodesInBatches(next.nodes, next.highlights, next.batchSize)
                            .then(next.resolve)
                            .catch(next.reject);
                    }
                    
                    if (pendingUpdate) {
                        pendingUpdate = false;
                        debounceHighlight(applyHighlights);
                    }
                }
            } catch (error) {
                isProcessing = false;
                reject(error);
            }
        }
        
        processBatch();
    });
}

// 优化：智能节点过滤
function shouldProcessNode(node) {
    if (!node || !node.parentNode) return false;
    
    // 检查是否已经是高亮节点
    if (node.parentNode.tagName === 'MARK') return false;
    
    const parent = node.parentNode;
    const style = window.getComputedStyle(parent);
    
    // 跳过隐藏元素
    if (style.display === 'none' || style.visibility === 'hidden' || 
        style.opacity === '0' || parent.offsetHeight === 0) return false;
    
    // 跳过特定元素
    const tagName = parent.tagName.toLowerCase();
    if (['script', 'style', 'textarea', 'input', 'select', 'option', 'mark', 'code', 'pre', 'svg', 'canvas'].includes(tagName)) {
        return false;
    }
    
    // 跳过空文本和纯空白文本
    const text = node.textContent.trim();
    if (!text) return false;
    
    // 检查当前网站
    const currentHost = window.location.hostname;
    
    // 排除飞书多维表格
    if (currentHost.includes('feishu.cn') || currentHost.includes('lark.com')) {
        // 排除多维表格相关元素
        if (window.location.href.includes('/bitable/') || 
            window.location.href.includes('/base/') ||
            document.querySelector('.bitable-embed-container')) {
            // 完全排除多维表格页面，防止空白屏幕问题
            return false;
        }
        
        // 此处可以添加更多飞书特定排除规则
        const parentDataset = parent.dataset || {};
        const parentClasses = parent.className || '';
        
        if (parentClasses.includes('sheet') || 
            parentClasses.includes('cell') || 
            parentClasses.includes('table-') || 
            parentClasses.includes('bitable') ||
            parentClasses.includes('editable-') ||
            parentDataset.role === 'grid' ||
            parentDataset.role === 'gridcell') {
            return false;
        }
    }
    
    // 排除常见的Web应用程序编辑区域，这些区域往往具有交互性
    const sensitiveClasses = [
        'editor', 'monaco-editor', 'CodeMirror', 'ace_editor', 
        'ql-editor', 'content-editable', 'ProseMirror', 'rich-text',
        'draft-js', 'medium-editor', 'tinymce', 'quill', 'slate-editor',
        'iframe-container', 'grid-', 'table-cell', 'data-grid', 'spreadsheet',
        'react-grid', 'ag-grid', 'handsontable', 'sheet-', 'excel-', 'pivot-'
    ];
    
    // 检查父元素的类名是否包含任何敏感类名
    if (parent.className && typeof parent.className === 'string') {
        for (const cls of sensitiveClasses) {
            if (parent.className.includes(cls)) {
                return false;
            }
        }
    }
    
    // 检查是否是编辑状态元素
    if (parent.contentEditable === 'true' || 
        parent.isContentEditable || 
        parent.getAttribute('role') === 'textbox' ||
        parent.getAttribute('role') === 'grid' ||
        parent.getAttribute('role') === 'gridcell' ||
        parent.getAttribute('role') === 'table') {
        return false;
    }
    
    // 检查是否是交互式元素
    const interactiveRoles = ['button', 'checkbox', 'combobox', 'menuitem', 'menuitemcheckbox', 
                           'menuitemradio', 'option', 'radio', 'slider', 'spinbutton', 
                           'switch', 'tab', 'tabpanel', 'textbox', 'treeitem'];
                           
    if (parent.getAttribute('role') && interactiveRoles.includes(parent.getAttribute('role'))) {
        return false;
    }
    
    // 特殊处理百度文库
    if (window.location.hostname.includes('wenku.baidu.com')) {
        // 检查是否在有效的内容区域内
        const isInValidContainer = WENKU_CONFIG.contentSelectors.some(selector => {
            return parent.closest(selector) !== null;
        });
        if (!isInValidContainer) return false;
        
        // 跳过不需要处理的元素
        const invalidClasses = ['reader-pic-item', 'reader-blank-item'];
        if (invalidClasses.some(cls => parent.classList.contains(cls))) {
            return false;
        }
    }
    
    // 扩大高亮应用范围，包括更多常见的内容区域
    const commonContentSelectors = [
        'main', 'article', 'section', '.content', '.main-content', 
        '.post-content', '.entry-content', '.article-content',
        '.page-content', '.text-content', '.body-content',
        'p', 'div', 'span', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6',
        'li', 'td', 'th', '.title', '.description'
    ];
    
    // 检查是否在常见内容区域内
    const isInContentArea = commonContentSelectors.some(selector => {
        try {
            return parent.matches(selector) || parent.closest(selector) !== null;
        } catch (e) {
            return false;
        }
    });
    
    // 如果不在内容区域内，但文本长度足够，也允许高亮
    if (!isInContentArea && text.length < 10) {
        return false;
    }
    
    return true;
}

// 优化：使用防抖处理高亮应用
function debounceHighlight(fn, delay = DEBOUNCE_DELAY) {
    if (processingTimeout) {
        clearTimeout(processingTimeout);
    }
    processingTimeout = setTimeout(() => {
        if (!isProcessing) {
            fn();
        }
        processingTimeout = null;
    }, currentSettings.performanceMode ? delay * 2 : delay);
}

// 添加节流函数
function throttle(func, limit) {
    let inThrottle;
    return function(...args) {
        if (!inThrottle) {
            func.apply(this, args);
            inThrottle = true;
            setTimeout(() => inThrottle = false, limit);
        }
    };
}

// 自动高亮处理
function handleAutoHighlight() {
    if (!currentSettings.autoHighlight) return;
    
    const selection = window.getSelection();
    if (!selection.toString().trim()) return;
    
    if (autoHighlightTimeout) {
        clearTimeout(autoHighlightTimeout);
    }
    
    autoHighlightTimeout = setTimeout(() => {
        chrome.runtime.sendMessage({
            action: 'addHighlight',
            text: selection.toString().trim()
        });
    }, 500);
}

// 添加网站兼容性检查
function checkSiteCompatibility() {
    const currentHost = window.location.hostname;
    const currentPath = window.location.pathname;
    const currentUrl = window.location.href;
    
    // 完全排除的网站列表（这些网站与插件不兼容）
    const fullyExcludedDomains = [
    { domain: 'feishu.cn', paths: ['/base/', '/bitable/'] },
    { domain: 'lark.com', paths: ['/base/', '/bitable/'] },
    { domain: 'shimo.im', paths: ['/sheet'] },
    { domain: 'airtable.com', paths: [] },
    { domain: 'notion.so', paths: [] },
    { domain: 'figma.com', paths: [] },
    { domain: 'canva.com', paths: [] },
    { domain: 'trello.com', paths: [] },
    { domain: 'miro.com', paths: [] },
    // 新增容易导致性能问题的网站
    { domain: 'docs.google.com', paths: [] },
    { domain: 'office.com', paths: [] },
    { domain: 'mail.google.com', paths: [] },
    { domain: 'outlook.com', paths: [] },
    { domain: 'yuque.com', paths: [] },
    { domain: 'processon.com', paths: [] },
    { domain: 'modao.cc', paths: [] }
];
    
    // 检查是否在完全排除的域名列表中
    for (const item of fullyExcludedDomains) {
        if (currentHost.includes(item.domain)) {
            // 如果指定了路径，检查当前路径是否匹配
            if (item.paths.length > 0) {
                for (const path of item.paths) {
                    if (currentUrl.includes(path)) {
                        console.debug('网站已排除:', currentHost, path);
                        return false;
                    }
                }
            } else {
                // 如果没有指定路径，则整个域名排除
                console.debug('网站已排除:', currentHost);
                return false;
            }
        }
    }
    
    // 检查特殊情况：CSDN代码编辑器、GitHub代码视图等
    if ((currentHost.includes('github.com') && currentPath.includes('/blob/')) ||
        (currentHost.includes('csdn.net') && document.querySelector('.editor_area')) ||
        document.querySelector('div[data-lexical-editor="true"]') || // Lexical编辑器
        document.querySelector('.monaco-editor') || // VS Code Web编辑器
        document.querySelector('.CodeMirror') || // CodeMirror编辑器
        document.querySelector('.ace_editor')) { // Ace编辑器
        console.debug('检测到代码编辑器，已禁用高亮');
        return false;
    }
    
    // 检查是否在表格应用中
    if (document.querySelector('.bitable-embed-container') ||
        document.querySelector('[data-sheet-id]') ||
        document.querySelector('.grid-container') ||
        document.querySelector('.ag-root-wrapper') ||
        document.querySelector('.handsontable')) {
        console.debug('检测到表格应用，已禁用高亮');
        return false;
    }
    
    return true;
}

// 主要的高亮应用函数
// 优化：性能监控
let lastPerformanceWarning = 0;
function checkPerformance() {
    const now = Date.now();
    if (now - lastPerformanceWarning < 60000) return; // 每分钟最多警告一次
    
    const perfData = window.performance.memory;
    if (perfData) {
        const usedMB = Math.round(perfData.usedJSHeapSize / 1024 / 1024);
        const totalMB = Math.round(perfData.totalJSHeapSize / 1024 / 1024);
        const limitMB = Math.round(perfData.jsHeapSizeLimit / 1024 / 1024);
        const usageRatio = perfData.usedJSHeapSize / perfData.jsHeapSizeLimit;
        
        console.debug('内存使用情况:', {
            used: usedMB + 'MB',
            total: totalMB + 'MB',
            limit: limitMB + 'MB',
            usage: Math.round(usageRatio * 100) + '%',
            cacheStats: { hits: cacheHits, misses: cacheMisses, hitRate: Math.round(cacheHits / (cacheHits + cacheMisses) * 100) + '%' }
        });
        
        if (usageRatio > 0.9) {
            currentSettings.performanceMode = true;
            cachedRegexps.clear();
            console.warn('内存使用过高，启用性能模式并清理缓存');
            lastPerformanceWarning = now;
        } else if (usageRatio > 0.8) {
            currentSettings.performanceMode = true;
            console.warn('内存使用较高，建议开启性能模式');
            lastPerformanceWarning = now;
        }
    }
    
    // 检查处理队列长度
    if (processingQueue.length > 5) {
        console.warn('处理队列过长，可能存在性能问题');
        processingQueue = processingQueue.slice(-3); // 只保留最新的3个任务
    }
}

function applyHighlights() {
    if (!isExtensionContextValid()) {
        console.debug('应用高亮时扩展上下文已失效，跳过操作');
        return;
    }
    
    checkPerformance(); // 添加性能检查
    
    // 检查网站兼容性
    if (!checkSiteCompatibility()) {
        console.debug('当前网站与插件不兼容，跳过高亮应用');
        return;
    }
    
    if (!highlightEnabled || !currentHighlights.length) {
        removeHighlights();
        return;
    }
    
    // 确保当前数据是最新的
    chrome.storage.local.get(['enabled', 'highlights', 'settings', 'currentHighlights'], function(data) {
        try {
            if (!isExtensionContextValid() || !checkSiteCompatibility()) return;
            
            // 更新本地变量以确保使用最新数据
            highlightEnabled = data.enabled !== undefined ? data.enabled : true;
            
            // 根据onlyCurrentTab设置选择正确的高亮数据源
            const settings = data.settings || {};
            if (settings.onlyCurrentTab) {
                // 在onlyCurrentTab模式下，优先使用currentHighlights
                // 这些数据由background.js或popup.js设置
                const newCurrentHighlights = data.currentHighlights ?? [];
                
                // 只有当数据真正发生变化时才更新
                if (JSON.stringify(currentHighlights) !== JSON.stringify(newCurrentHighlights)) {
                    currentHighlights = newCurrentHighlights;
                    console.debug('onlyCurrentTab模式：更新currentHighlights:', currentHighlights.length);
                } else {
                    console.debug('onlyCurrentTab模式：currentHighlights无变化:', currentHighlights.length);
                }
            } else {
                // 使用全局高亮数据
                const newGlobalHighlights = data.highlights?.default ?? [];
                
                // 只有当数据真正发生变化时才更新
                if (JSON.stringify(currentHighlights) !== JSON.stringify(newGlobalHighlights)) {
                    currentHighlights = newGlobalHighlights;
                    console.debug('全局模式：更新default高亮词:', currentHighlights.length);
                } else {
                    console.debug('全局模式：default高亮词无变化:', currentHighlights.length);
                }
            }
            
            currentSettings = {
                ...currentSettings,
                ...settings,
                autoHighlight: settings?.autoHighlight ?? true,
                performanceMode: settings?.performanceMode !== false
            };
            
            // 继续执行高亮应用逻辑
            continueApplyHighlights();
            
        } catch (err) {
            console.error('获取存储数据时出错:', err);
        }
    });
}

// 将高亮应用的核心逻辑提取为独立函数
function continueApplyHighlights() {
    try {
        if (!highlightEnabled || !currentHighlights.length) {
            removeHighlights();
            return;
        }
        
        // 先移除所有现有高亮，确保重新应用时使用最新的设置
        removeHighlights();
        
        // 智能清理正则表达式缓存
        if (cachedRegexps.size > MAX_CACHE_SIZE * 0.8) {
            const keysToDelete = Array.from(cachedRegexps.keys()).slice(0, Math.floor(cachedRegexps.size / 2));
            keysToDelete.forEach(k => cachedRegexps.delete(k));
        }
        
        // 获取文本节点
        const walker = document.createTreeWalker(
            document.body,
            NodeFilter.SHOW_TEXT,
            {
                acceptNode: function(node) {
                    return shouldProcessNode(node) ? 
                        NodeFilter.FILTER_ACCEPT : 
                        NodeFilter.FILTER_REJECT;
                }
            }
        );
        
        const nodes = [];
        let currentNode;
        
        // 收集所有符合条件的节点，不限制数量以确保完整高亮
        while ((currentNode = walker.nextNode())) {
            nodes.push(currentNode);
        }
        
        console.debug(`收集到 ${nodes.length} 个文本节点`);
        
        // 对每个节点应用所有高亮
        processNodesInBatches(nodes, currentHighlights)
            .then(processedCount => {
                console.debug(`处理完成，共处理 ${processedCount} 个节点`);
                // 处理iframe内容（延迟执行）
                setTimeout(() => processIframes(), 100);
            })
            .catch(error => {
                console.error('处理节点时出错:', error);
            });
        
        // 记录应用完成的时间，用于调试
        console.debug('高亮应用完成，时间：', new Date().toISOString(), '高亮词数量：', currentHighlights.length);
    } catch (error) {
        console.error('应用高亮时发生错误:', error);
    }
}

// 优化：处理iframe内容
function processIframes() {
    // 处理所有iframe
    WENKU_CONFIG.iframeSelectors.forEach(selector => {
        const iframes = document.querySelectorAll(selector);
        iframes.forEach(iframe => {
            try {
                if (iframe.contentDocument) {
                    initializeForFrame(iframe.contentDocument);
                }
            } catch (e) {
                console.warn('无法访问iframe内容:', e);
            }
        });
    });
}

// 优化：初始化frame内容
function initializeForFrame(doc) {
    const walker = document.createTreeWalker(
        doc.body,
        NodeFilter.SHOW_TEXT,
        {
            acceptNode: function(node) {
                return shouldProcessNode(node) ? 
                    NodeFilter.FILTER_ACCEPT : 
                    NodeFilter.FILTER_REJECT;
            }
        }
    );

    const textNodes = [];
    let node;
    while (node = walker.nextNode()) {
        textNodes.push(node);
    }

    if (textNodes.length > 0) {
        processNodesInBatches(textNodes, currentHighlights);
    }
}

// 优化：监听动态内容
function observeWenkuContent() {
    const observer = new MutationObserver((mutations) => {
        let shouldUpdate = false;
        
        for (const mutation of mutations) {
            // 检查是否有新的文本节点添加
            if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
                shouldUpdate = true;
                break;
            }
            // 检查文本内容是否改变
            if (mutation.type === 'characterData') {
                shouldUpdate = true;
                break;
            }
        }

        if (shouldUpdate) {
            // 使用requestAnimationFrame代替debounce，减少延迟
            requestAnimationFrame(() => {
                if (!isExtensionContextValid() || !checkSiteCompatibility()) return;
                applyHighlights();
            });
        }
    });

    // 监听文库内容区域
    WENKU_CONFIG.contentSelectors.forEach(selector => {
        const elements = document.querySelectorAll(selector);
        elements.forEach(element => {
            observer.observe(element, WENKU_CONFIG.observerConfig);
        });
    });

    // 监听iframe加载
    WENKU_CONFIG.iframeSelectors.forEach(selector => {
        const iframes = document.querySelectorAll(selector);
        iframes.forEach(iframe => {
            iframe.addEventListener('load', () => {
                try {
                    if (iframe.contentDocument) {
                        initializeForFrame(iframe.contentDocument);
                        observer.observe(iframe.contentDocument.body, WENKU_CONFIG.observerConfig);
                    }
                } catch (e) {
                    console.warn('无法访问iframe内容:', e);
                }
            });
        });
    });
}

// 修改初始化函数
function initialize() {
    if (!isExtensionContextValid()) {
        console.debug('初始化时扩展上下文已失效，跳过初始化');
        return;
    }
    
    // 首先检查网站兼容性
    if (!checkSiteCompatibility()) {
        console.debug('当前网站与插件不兼容，已禁用高亮功能');
        return;
    }
    
    // 使用local而不是sync存储
    chrome.storage.local.get(['enabled', 'highlights', 'settings', 'currentHighlights'], function(data) {
        try {
            if (!isExtensionContextValid()) return;
            
            highlightEnabled = data.enabled !== undefined ? data.enabled : true;
            
            // 获取当前设置
            currentSettings = {
                ...currentSettings,
                ...data.settings,
                autoHighlight: data.settings?.autoHighlight ?? true, // 确保默认为true
                globalMatch: true,
                processAll: true,
                multipleMatches: true,
                maintainAllHighlights: true
            };
            
            // 根据onlyCurrentTab设置选择正确的高亮词来源
            if (currentSettings.onlyCurrentTab) {
                // 在onlyCurrentTab模式下，使用currentHighlights
                currentHighlights = data.currentHighlights ?? [];
                console.debug('使用当前标签页高亮词:', currentHighlights.length);
            } else {
                // 否则使用全局高亮词
                currentHighlights = data.highlights?.default ?? [];
                console.debug('使用全局高亮词:', currentHighlights.length);
            }
            
            // 确保在初始化时应用高亮
            if (highlightEnabled && currentSettings.autoHighlight && currentHighlights.length > 0) {
                // 使用requestAnimationFrame确保在下一帧渲染前应用高亮
                requestAnimationFrame(() => {
                    if (!isExtensionContextValid() || !checkSiteCompatibility()) return;
                    
                    // 再次检查网站兼容性
                    removeHighlights(); // 先清除现有高亮
                    applyHighlights(); // 重新应用高亮
                });
            }
            
            // 是否启用自动高亮功能根据设置决定
            if (currentSettings.autoHighlight) {
                document.addEventListener('mouseup', handleAutoHighlight);
                document.addEventListener('keyup', handleAutoHighlight);
            }
        } catch (error) {
            console.error('初始化函数内部出错:', error);
        }
    });

    // 添加百度文库支持
    if (window.location.hostname.includes('wenku.baidu.com')) {
        // 初始化iframe处理
        processIframes();
        
        // 监听动态内容
        observeWenkuContent();
        
        // 监听URL变化（处理单页应用）
        let lastUrl = location.href;
        new MutationObserver(() => {
            if (!isExtensionContextValid() || !checkSiteCompatibility()) return;
            
            const url = location.href;
            if (url !== lastUrl) {
                lastUrl = url;
                requestAnimationFrame(() => {
                    if (!isExtensionContextValid() || !checkSiteCompatibility()) return;
                    processIframes();
                    applyHighlights();
                });
            }
        }).observe(document, { subtree: true, childList: true });
    }
}

// 监听消息
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (!isExtensionContextValid()) {
        console.debug('接收消息时扩展上下文已失效，跳过处理');
        if (sendResponse) {
            sendResponse({ success: false, error: 'Extension context invalidated' });
        }
        return true;
    }
    
    try {
        // 处理ping消息不检查站点兼容性
        if (request.action === 'ping') {
            // 回复，表明内容脚本已准备就绪
            if (sendResponse) {
                sendResponse({ 
                    success: true, 
                    scriptReady: true, 
                    documentState: document.readyState,
                    siteCompatible: checkSiteCompatibility()
                });
            }
            return true;
        }
        
        // 对于其他所有消息，检查网站兼容性
        if (!checkSiteCompatibility()) {
            console.debug('当前网站与插件不兼容，跳过消息处理:', request.action);
            if (sendResponse) {
                sendResponse({ 
                    success: false, 
                    error: 'Site incompatible', 
                    siteCompatible: false 
                });
            }
            return true;
        }
        
        switch (request.action) {
            case 'updateHighlights':
                highlightEnabled = request.enabled;
                
                // 完全替换当前高亮列表，确保使用最新数据
                if (request.highlights) {
                    currentHighlights = request.highlights;
                    
                    // 如果是onlyCurrentTab模式，更新currentHighlights存储
                    if (currentSettings.onlyCurrentTab || (request.settings && request.settings.onlyCurrentTab)) {
                        chrome.storage.local.set({ currentHighlights: currentHighlights }, function() {
                            console.debug('已更新当前标签页高亮词到storage:', currentHighlights.length);
                        });
                    }
                }
                
                if (request.settings) {
                    const oldCaseSensitive = currentSettings.caseSensitive;
                    const oldOnlyCurrentTab = currentSettings.onlyCurrentTab;
                    currentSettings = { ...currentSettings, ...request.settings };
                    
                    // 如果大小写设置发生变化，清除正则表达式缓存
                    if (oldCaseSensitive !== currentSettings.caseSensitive) {
                        cachedRegexps.clear();
                    }
                    
                    // 如果onlyCurrentTab设置发生变化，可能需要重新加载高亮词
                    if (oldOnlyCurrentTab !== currentSettings.onlyCurrentTab) {
                        console.debug('onlyCurrentTab设置已更改，重新加载高亮词');
                        // 这里不需要额外处理，因为highlights已经在请求中提供
                    }
                }
                
                // 先移除所有现有高亮，确保重新应用时使用最新的设置
                removeHighlights();
                
                // 根据自动高亮设置和force参数决定是否应用高亮
                if (highlightEnabled && (currentSettings.autoHighlight || request.force)) {
                    // 确保页面上没有残留的高亮
                    removeHighlights();
                    
                    // 立即应用高亮，不使用延迟以避免闪烁
                    try {
                        // 使用requestAnimationFrame确保在下一帧渲染前应用高亮
                        requestAnimationFrame(() => {
                            if (!isExtensionContextValid() || !checkSiteCompatibility()) {
                                if (sendResponse) {
                                    sendResponse({ 
                                        success: false, 
                                        error: 'Extension context invalidated or site incompatible',
                                        siteCompatible: checkSiteCompatibility()
                                    });
                                }
                                return;
                            }
                            
                            try {
                                applyHighlights();
                                // 操作完成后发送响应
                                if (sendResponse) {
                                    sendResponse({ success: true, action: 'updateHighlights' });
                                }
                            } catch (err) {
                                console.error('应用高亮时出错:', err);
                                if (sendResponse) {
                                    sendResponse({ 
                                        success: false, 
                                        error: '应用高亮时出错: ' + (err.message || JSON.stringify(err)) 
                                    });
                                }
                            }
                        });
                    } catch (err) {
                        console.error('请求动画帧时出错:', err);
                        if (sendResponse) {
                            sendResponse({ 
                                success: false, 
                                error: '请求动画帧时出错: ' + (err.message || JSON.stringify(err)) 
                            });
                        }
                    }
                } else {
                    if (sendResponse) {
                        sendResponse({ success: true, action: 'updateHighlights', noHighlights: true });
                    }
                }
                break;
                
            case 'updateSetting':
                // 处理单个设置更新
                try {
                    const oldValue = currentSettings[request.setting];
                    currentSettings[request.setting] = request.value;
                    
                    // 如果是大小写设置变化，清除缓存
                    if (request.setting === 'caseSensitive' && oldValue !== request.value) {
                        cachedRegexps.clear();
                        if (highlightEnabled) {
                            removeHighlights();
                            requestAnimationFrame(() => {
                                if (!isExtensionContextValid() || !checkSiteCompatibility()) return;
                                applyHighlights();
                            });
                        }
                    }
                    
                    if (sendResponse) {
                        sendResponse({ success: true, action: 'updateSetting', setting: request.setting });
                    }
                } catch (err) {
                    console.error('更新单个设置时出错:', err);
                    if (sendResponse) {
                        sendResponse({ 
                            success: false, 
                            error: '更新单个设置时出错: ' + (err.message || JSON.stringify(err))
                        });
                    }
                }
                break;
                
            case 'updateSettings':
                const oldAutoHighlight = currentSettings.autoHighlight;
                const oldCaseSensitive = currentSettings.caseSensitive;
                
                try {
                    currentSettings = { ...currentSettings, ...request.settings };
                    
                    // 如果大小写设置发生变化，清除正则表达式缓存并立即重新应用高亮
                    if (oldCaseSensitive !== currentSettings.caseSensitive) {
                        cachedRegexps.clear();
                        // 无论 autoHighlight 设置如何，都立即重新应用高亮
                        if (highlightEnabled) {
                            removeHighlights();
                            requestAnimationFrame(() => {
                                if (!isExtensionContextValid() || !checkSiteCompatibility()) return;
                                
                                try {
                                    applyHighlights();
                                    if (sendResponse) {
                                        sendResponse({ 
                                            success: true, 
                                            action: 'updateSettings', 
                                            changed: 'caseSensitive' 
                                        });
                                    }
                                } catch (err) {
                                    console.error('更新设置后重新应用高亮时出错:', err);
                                    if (sendResponse) {
                                        sendResponse({ 
                                            success: false, 
                                            error: '更新设置后重新应用高亮时出错: ' + (err.message || JSON.stringify(err)) 
                                        });
                                    }
                                }
                            });
                            // 不立即返回，等待异步响应
                        } else {
                            if (sendResponse) {
                                sendResponse({ 
                                    success: true, 
                                    action: 'updateSettings', 
                                    changed: 'caseSensitive', 
                                    noApply: true 
                                });
                            }
                        }
                    } else {
                        if (sendResponse) {
                            sendResponse({ success: true, action: 'updateSettings' });
                        }
                    }
                    
                    // 处理自动高亮事件监听器
                    if (!currentSettings.autoHighlight) {
                        document.removeEventListener('mouseup', handleAutoHighlight);
                        document.removeEventListener('keyup', handleAutoHighlight);
                    } else if (!oldAutoHighlight) {
                        document.addEventListener('mouseup', handleAutoHighlight);
                        document.addEventListener('keyup', handleAutoHighlight);
                    }
                } catch (err) {
                    console.error('更新设置时出错:', err);
                    if (sendResponse) {
                        sendResponse({ 
                            success: false, 
                            error: '更新设置时出错: ' + (err.message || JSON.stringify(err))
                        });
                    }
                }
                break;
                
            case 'removeHighlights':
                try {
                    if (request.highlightTexts && !request.force) {
                        // 移除指定的高亮词
                        currentHighlights = currentHighlights.filter(h => 
                            !request.highlightTexts.includes(h.text)
                        );
                        
                        // 移除DOM中的高亮标记
                        removeHighlights();
                        
                        // 如果启用了自动高亮，重新应用其余高亮
                        if (currentSettings.autoHighlight) {
                            applyHighlights();
                        }
                    } else {
                        // 彻底移除所有高亮
                        currentHighlights = [];
                        removeHighlights();
                    }
                    
                    if (sendResponse) {
                        sendResponse({ success: true, action: 'removeHighlights' });
                    }
                } catch (err) {
                    console.error('移除高亮时出错:', err);
                    if (sendResponse) {
                        sendResponse({ 
                            success: false, 
                            error: '移除高亮时出错: ' + (err.message || JSON.stringify(err))
                        });
                    }
                }
                break;
                
            default:
                if (sendResponse) {
                    sendResponse({ success: false, error: 'Unknown action: ' + request.action });
                }
                console.debug('收到未知操作:', request.action);
        }
    } catch (error) {
        // 捕获所有其他错误
        const errorMsg = error.message || JSON.stringify(error);
        console.error('处理消息时出错:', errorMsg, error);
        if (sendResponse) {
            sendResponse({ 
                success: false, 
                error: '处理消息时出错: ' + errorMsg,
                stack: error.stack
            });
        }
    }
    
    // 返回true表示异步发送响应
    return true;
});

// 确保在DOMContentLoaded和页面完全加载后初始化
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
        try {
            initialize();
            console.debug('内容脚本在DOMContentLoaded时初始化完成');
        } catch (error) {
            console.error('初始化过程中出错:', error);
        }
    });
} else {
    // 页面已经加载完成，立即初始化
    try {
        initialize();
        console.debug('内容脚本立即初始化完成');
    } catch (error) {
        console.error('初始化过程中出错:', error);
    }
}

// 添加window.onload事件处理，确保页面完全加载后也应用高亮
window.addEventListener('load', function() {
    try {
        // 通知后台脚本，内容脚本已准备就绪
        chrome.runtime.sendMessage({ action: 'contentScriptReady' }, (response) => {
            if (chrome.runtime.lastError) {
                console.debug('通知后台脚本时出错:', chrome.runtime.lastError);
                return;
            }
            console.debug('内容脚本通知后台已准备就绪');
        });
        
        // 使用requestAnimationFrame确保在下一帧渲染前应用高亮
        requestAnimationFrame(() => {
            // 重新获取数据并应用高亮
            chrome.storage.local.get(['enabled', 'highlights', 'settings'], function(data) {
                highlightEnabled = data.enabled !== undefined ? data.enabled : true;
                currentHighlights = data.highlights?.default ?? [];
                currentSettings = {
                    ...currentSettings,
                    ...data.settings,
                    autoHighlight: data.settings?.autoHighlight ?? true
                };
                
                if (highlightEnabled && currentSettings.autoHighlight && currentHighlights.length > 0) {
                    removeHighlights(); // 先清除现有高亮
                    applyHighlights(); // 重新应用高亮
                    console.debug('页面完全加载后重新应用了高亮');
                }
            });
        }, 1000);
    } catch (error) {
        console.error('加载事件处理时出错:', error);
    }
});

// 移除默认的事件监听器，让它们通过 initialize 和 updateSettings 来管理
document.removeEventListener('mouseup', handleAutoHighlight);
document.removeEventListener('keyup', handleAutoHighlight);

// 监听DOM变化，处理动态添加的内容
const observer = new MutationObserver(throttle((mutations) => {
    // 避免在不可交互页面上执行不必要的处理
    if (!isExtensionContextValid() || !highlightEnabled || !currentHighlights.length) {
        return;
    }
    
    // 检查当前域名是否需要排除
    const currentHost = window.location.hostname;
    const excludedDomains = [
        'feishu.cn', 'lark.com', 'shimo.im', 'yuque.com', 'notion.so', 
        'figma.com', 'canva.com', 'miro.com', 'trello.com', 
        'airtable.com', 'monday.com', 'asana.com'
    ];
    
    // 如果当前网站在排除列表中，不进行处理
    for (const domain of excludedDomains) {
        if (currentHost.includes(domain)) {
            // 如果在指定的应用页面，不处理DOM变化
            if (window.location.href.includes('/base/') || 
                window.location.href.includes('/bitable/') ||
                window.location.href.includes('/file/') ||
                window.location.href.includes('/board/') ||
                window.location.href.includes('/table/') ||
                window.location.href.includes('/sheet/')) {
                return;
            }
        }
    }
    
    // 性能优化：使用节流方式处理大量DOM变化
    let shouldUpdate = false;
    let hasDynamicContent = false;
    let addedTextNodes = 0;
    
    // 智能分析变化
    for (const mutation of mutations) {
        // 检查是否有新的文本节点添加
        if (mutation.type === 'childList') {
            // 检查是否有任何新添加的节点
            if (mutation.addedNodes.length > 0) {
                // 只处理可能包含文本内容的节点
                for (const node of mutation.addedNodes) {
                    // 检查是否是可能包含文本的元素节点
                    if (node.nodeType === Node.TEXT_NODE) {
                        addedTextNodes++;
                        shouldUpdate = true;
                    } else if (node.nodeType === Node.ELEMENT_NODE && 
                              node.textContent && 
                              node.textContent.trim().length > 0 &&
                              !isInteractiveElement(node)) {
                        shouldUpdate = true;
                        hasDynamicContent = true;
                    }
                }
            }
        }
        
        // 检查文本内容是否变化（并且不是用户输入）
        if (mutation.type === 'characterData' && 
            mutation.target.textContent && 
            mutation.target.textContent.trim().length > 0 &&
            !isInteractiveElement(mutation.target)) {
            shouldUpdate = true;
            break;
        }
        
        if (shouldUpdate) break;
    }
    
    // 如果需要更新且不是在动态内容区域，使用防抖函数更新高亮
    if (shouldUpdate) {
        // 根据变化量调整延迟
        const delay = addedTextNodes > 10 ? 1000 : (hasDynamicContent ? 500 : DEBOUNCE_DELAY);
        debounceHighlight(applyHighlights, delay);
    }
}, 200)); // 节流200ms

// 辅助函数：检查元素是否是交互式元素
function isInteractiveElement(node) {
    if (!node) return false;
    
    // 如果是文本节点，检查其父元素
    if (node.nodeType === Node.TEXT_NODE) {
        return node.parentNode ? isInteractiveElement(node.parentNode) : false;
    }
    
    // 检查元素标签
    if (node.tagName) {
        const tagName = node.tagName.toLowerCase();
        if (['input', 'textarea', 'select', 'option', 'button', 'iframe', 'form'].includes(tagName)) {
            return true;
        }
    }
    
    // 检查元素属性
    if (node.contentEditable === 'true' || 
        node.isContentEditable || 
        node.getAttribute('role') === 'textbox' || 
        node.getAttribute('role') === 'grid' ||
        node.getAttribute('role') === 'gridcell') {
        return true;
    }
    
    // 检查类名是否包含交互式类名
    const interactiveClasses = [
        'editor', 'monaco-editor', 'CodeMirror', 'ace_editor', 
        'ql-editor', 'content-editable', 'ProseMirror', 'rich-text',
        'draft-js', 'medium-editor', 'tinymce', 'quill', 'slate-editor',
        'sheet-', 'cell-', 'grid-', 'table-', 'data-grid', 'spreadsheet'
    ];
    
    if (node.className && typeof node.className === 'string') {
        for (const cls of interactiveClasses) {
            if (node.className.includes(cls)) {
                return true;
            }
        }
    }
    
    // 递归检查父节点
    return node.parentNode ? isInteractiveElement(node.parentNode) : false;
}

try {
    // 确保body存在
    if (document.body) {
        // 使用更保守的配置，减少CPU和内存使用
        observer.observe(document.body, {
            childList: true,
            subtree: true,
            characterData: true,
            attributeFilter: ['style', 'class'], // 只监听可能影响显示的属性变化
            characterDataOldValue: false, // 不需要存储旧值
        });
    } else {
        // 如果body不存在，等待它加载完成
        document.addEventListener('DOMContentLoaded', () => {
            observer.observe(document.body, {
                childList: true,
                subtree: true,
                characterData: true,
                attributeFilter: ['style', 'class'],
                characterDataOldValue: false,
            });
        });
    }
} catch (error) {
    console.error('初始化DOM观察器时发生错误:', error);
}