document.addEventListener('DOMContentLoaded', function() {
    // 优化的节流函数
    function throttle(func, limit) {
        let inThrottle;
        let lastArgs;
        return function(...args) {
            lastArgs = args;
            if (!inThrottle) {
                func.apply(this, args);
                inThrottle = true;
                setTimeout(() => {
                    inThrottle = false;
                    // 如果在等待期间有新的调用，执行最后一次
                    if (lastArgs !== args) {
                        func.apply(this, lastArgs);
                    }
                }, limit);
            }
        }
    }
    
    // 添加转义正则表达式特殊字符的函数
    function escapeRegExp(string) {
        return string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
    }

    // 优化的updateHighlights函数
    function updateHighlights(force = false) {
        chrome.tabs.query({active: true, currentWindow: true}, function(tabs) {
            if (!tabs || !tabs.length || !tabs[0] || !tabs[0].id) {
                console.debug('没有可用的活动标签页');
                return;
            }
            
            const activeTab = tabs[0];
            
            // 只发送消息到http/https页面
            if (!activeTab.url || !activeTab.url.match(/^https?:\/\//)) {
                console.debug('不支持当前页面类型:', activeTab.url);
                return;
            }
            
            chrome.storage.local.get(['highlights', 'settings'], function(data) {
                const highlights = data.highlights?.default ?? [];
                const settings = {
                    renderMode: document.getElementById('renderMode')?.value || 'stable',
                    highlightDelay: parseInt(document.getElementById('highlightDelay')?.value || '800'),
                    useHardwareAcceleration: document.getElementById('useHardwareAcceleration')?.checked || true,
                    usePositionLocking: document.getElementById('usePositionLocking')?.checked || true,
                    batchSize: 25, // 减少批处理大小
                    processingInterval: 150, // 增加处理间隔
                    processAll: false, // 减少处理量
                    globalMatch: data.settings?.globalMatch !== false,
                    reprocessExisting: false, // 减少重复处理
                    multipleMatches: true,
                    maintainAllHighlights: false, // 减少维护开销
                    performanceMode: true // 启用性能模式
                };

                try {
                    // 直接更新高亮，避免先移除再添加的双重操作
                    chrome.tabs.sendMessage(activeTab.id, {
                        action: 'updateHighlights',
                        force: force,
                        highlights: highlights.map(h => ({
                            ...h,
                            globalMatch: settings.globalMatch,
                            processAll: false,
                            multipleMatches: true,
                            maintainAllHighlights: false
                        })),
                        settings: settings
                    }, function(response) {
                        // 检查是否收到响应
                        if (chrome.runtime.lastError) {
                            console.debug('更新高亮失败:', chrome.runtime.lastError);
                        }
                    });
                } catch (error) {
                    console.error('发送高亮消息时出错:', error);
                }
            });
        });
    }

    // 创建节流版本的updateHighlights，增加延迟减少频繁调用
    const throttledUpdate = throttle(updateHighlights, 2000); // 增加到2秒

    // Add settingsHtml
    const settingsHtml = `
        <div class="header">
            <h2>文本高亮工具</h2>
            <label class="toggle-switch-label">
                <span class="toggle-switch">
                    <input type="checkbox" id="enableHighlight">
                    <span class="toggle-slider"></span>
                </span>
            </label>
        </div>
        <div class="content-wrapper">
            <div class="tab-pane active" id="home">
                <div class="main-content">
                    <div class="settings-group top-settings">
                        <div class="settings-row">
                            <label>
                                <input type="checkbox" id="autoHighlight">
                                自动高亮
                            </label>
                            <label>
                                <input type="checkbox" id="caseSensitive">
                                区分大小写
                            </label>
                        </div>
                        <div class="settings-row">
                            <label>
                                <input type="checkbox" id="globalMatch">
                                全局匹配
                            </label>
                            <label>
                                <input type="checkbox" id="onlyCurrentTab">
                                只在当前标签页应用
                            </label>
                        </div>
                    </div>
                    <div class="input-group">
                        <input type="text" id="highlightWord" placeholder="输入要高亮的词语">
                        <input type="color" id="highlightColor" value="#ffeb3b">
                        <button id="addWord">添加</button>
                    </div>
                    <div class="import-group">
                        <button class="btn btn-gray" id="deleteSelected" disabled>删除所选</button>
                        <button class="btn btn-danger" id="clearAll">清除全部</button>
                        <input type="file" id="fileInput" accept=".txt" style="display: none">
                        <button class="btn btn-primary" id="importBtn">导入</button>
                        <button class="btn btn-secondary" id="exportBtn">导出</button>
                    </div>
                    <div class="search-group">
                        <span class="search-icon">🔍</span>
                        <input type="text" id="searchHighlight" placeholder="搜索高亮词...">
                    </div>
                    <div class="highlight-header">
                        <div class="highlight-list-title">当前高亮词</div>
                        <label class="select-all-label">
                            <input type="checkbox" id="selectAll">
                            <span class="checkmark"></span>
                            <span class="label-text">全选</span>
                        </label>
                    </div>
                    <div class="highlight-container" id="highlightContainer"></div>
                </div>
            </div>
            <div class="tab-pane" id="settings">
                <div class="settings-content">
                    <div class="settings-section">
                        <h3>渲染设置</h3>
                        <div class="setting-item">
                            <label>高亮延迟(ms)</label>
                            <input type="number" id="highlightDelay" min="0" max="1000" step="50" value="800">
                            <p class="setting-desc">较高的延迟可以减少闪烁，但会降低响应速度</p>
                        </div>
                        <div class="setting-item">
                            <label>渲染模式</label>
                            <select id="renderMode">
                                <option value="stable">稳定模式（推荐）</option>
                                <option value="fast">快速模式</option>
                            </select>
                            <p class="setting-desc">稳定模式优先考虑显示稳定性，快速模式优先考虑响应速度</p>
                        </div>
                    </div>
                    <div class="settings-section">
                        <h3>高级设置</h3>
                        <div class="setting-item">
                            <label class="switch-label">
                                <input type="checkbox" id="useHardwareAcceleration">
                                <span class="switch-text">启用硬件加速</span>
                            </label>
                            <p class="setting-desc">可能提高性能，但在某些设备上可能导致问题</p>
                        </div>
                        <div class="setting-item">
                            <label class="switch-label">
                                <input type="checkbox" id="usePositionLocking">
                                <span class="switch-text">启用位置锁定</span>
                            </label>
                            <p class="setting-desc">减少高亮位置抖动，但可能影响滚动性能</p>
                        </div>
                    </div>
                </div>
            </div>
            <div class="tab-pane" id="about">
                <div class="about-content">
                    <div class="about-section">
                        <div class="about-item">
                            <h4>版本信息</h4>
                            <p>当前版本：2.0.6</p>
                            <p>更新日期：2025-06-25</p>
                        </div>
                        <div class="about-item">
                            <h4>开发者信息</h4>
                            <p>作者：JiaoTang</p>
                            <p>联系邮箱：highlight@kwaie.cn</p>
                        </div>
                        <div class="about-item">
                            <h4>更新日志</h4>
                            <div class="doc-buttons">
                                <button class="btn" id="updateLog">查看更新日志</button>
                            </div>
                        </div>
                        <div class="about-item">
                            <h4>文档</h4>
                            <div class="doc-buttons">
                                <button class="btn" id="privacyPolicy">隐私政策</button>
                                <button class="btn" id="permissionList">权限清单</button>
                            </div>
                        </div>
                        <div class="about-item">
                            <h4>建议与反馈</h4>
                            <div class="doc-buttons">
                                <button class="btn" id="feedbackBtn">提交反馈</button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <div class="bottom-nav">
            <button class="nav-tab active" data-tab="home">首页</button>
            <button class="nav-tab" data-tab="settings">设置</button>
            <button class="nav-tab" data-tab="about">关于</button>
        </div>
    `;

    // Insert settings group at the beginning of body
    document.body.innerHTML = settingsHtml;

    // Get references to all controls
    const enableHighlight = document.getElementById('enableHighlight');
    const autoHighlight = document.getElementById('autoHighlight');
    const caseSensitive = document.getElementById('caseSensitive');
    const globalMatch = document.getElementById('globalMatch');
    const onlyCurrentTab = document.getElementById('onlyCurrentTab');
    const highlightWord = document.getElementById('highlightWord');
    const highlightColor = document.getElementById('highlightColor');
    const addWordBtn = document.getElementById('addWord');
    const fileInput = document.getElementById('fileInput');
    const importBtn = document.getElementById('importBtn');
    const exportBtn = document.getElementById('exportBtn');
    const highlightContainer = document.getElementById('highlightContainer');
    const selectAll = document.getElementById('selectAll');
    const deleteSelected = document.getElementById('deleteSelected');
    const clearAll = document.getElementById('clearAll');

    // Add updateHighlightList function
    function updateHighlightList(highlights) {
        const container = document.getElementById('highlightContainer');
        const searchTerm = document.getElementById('searchHighlight')?.value?.toLowerCase().trim() || '';
        
        if (!container) return;

        // 清空容器
        container.innerHTML = '';

        // 确保highlights是数组
        if (!Array.isArray(highlights)) {
            highlights = [];
        }

        // 过滤高亮词
        const displayHighlights = searchTerm
            ? highlights.filter(h => h.text.toLowerCase().includes(searchTerm))
            : highlights;
            
        // 更新高亮词数量显示
        updateHighlightCountDisplay(highlights.length, displayHighlights.length, 0);

        // 如果没有高亮词，显示提示
        if (displayHighlights.length === 0) {
            const noResults = document.createElement('div');
            noResults.className = 'no-results';
            noResults.innerHTML = searchTerm
                ? `未找到包含 "${searchTerm}" 的高亮词`
                : '暂无高亮词';
            container.appendChild(noResults);
            return;
        }

        // 创建并添加高亮词元素
        displayHighlights.forEach(highlight => {
            const div = document.createElement('div');
            div.className = 'highlight-item';
            div.dataset.id = highlight.id;
            
            // 使文本内搜索词高亮显示（如果有搜索）
            let displayText = highlight.text;
            if (searchTerm) {
                const regex = new RegExp(`(${escapeRegExp(searchTerm)})`, 'gi');
                displayText = displayText.replace(regex, '<mark>$1</mark>');
            }

            div.innerHTML = `
                <div class="word-info">
                    <input type="checkbox" class="checkbox">
                    <div class="color-preview" style="background-color: ${highlight.color}" title="点击修改颜色"></div>
                    <span class="text">${displayText}</span>
                    <div class="match-settings">
                        ${highlight.caseSensitive ? '<span class="match-tag fuzzy">区分大小写</span>' : ''}
                        ${highlight.globalMatch ? '<span class="match-tag whole">全局匹配</span>' : ''}
                    </div>
                </div>
                <div class="controls">
                    <button class="control-btn" title="编辑">✎</button>
                    <button class="control-btn delete-btn" title="删除">×</button>
                </div>
            `;

            // 添加事件监听器
            const checkbox = div.querySelector('.checkbox');
            checkbox.addEventListener('change', updateDeleteButtonState);

            const colorPreview = div.querySelector('.color-preview');
            colorPreview.addEventListener('click', () => {
                const input = document.createElement('input');
                input.type = 'color';
                input.value = highlight.color;
                input.addEventListener('change', function() {
                    updateHighlightColor(highlight, this.value);
                });
                input.click();
            });

            const editBtn = div.querySelector('.control-btn');
            editBtn.onclick = () => {
                const newWord = prompt('编辑高亮词:', highlight.text);
                if (newWord && newWord.trim() && newWord !== highlight.text) {
                    editHighlight(highlight.id, newWord.trim());
                }
            };

            const deleteBtn = div.querySelector('.delete-btn');
            deleteBtn.onclick = () => deleteHighlight(highlight.id);

            container.appendChild(div);
        });

        updateDeleteButtonState();
    }

    // Add updateHighlightColor function
    async function updateHighlightColor(highlight, newColor) {
        try {
            const data = await new Promise(resolve => {
                chrome.storage.local.get(['highlights', 'settings'], resolve);
            });
            
            const highlights = data.highlights ?? {};
            highlights.default = highlights.default ?? [];
            highlights.tabs = highlights.tabs ?? {};
            const settings = data.settings ?? {};
            
            let targetHighlights = [];
            let index = -1;
            
            if (settings.onlyCurrentTab) {
                // 修改当前标签页的高亮词颜色
                const tabs = await new Promise(resolve => {
                    chrome.tabs.query({active: true, currentWindow: true}, resolve);
                });
                
                if (tabs && tabs.length > 0) {
                    const currentTabId = tabs[0].id.toString();
                    highlights.tabs[currentTabId] = highlights.tabs[currentTabId] ?? [];
                    targetHighlights = highlights.tabs[currentTabId];
                    index = targetHighlights.findIndex(h => h.id === highlight.id);
                }
            } else {
                // 修改全局高亮词颜色
                targetHighlights = highlights.default;
                index = targetHighlights.findIndex(h => h.id === highlight.id);
            }
            
            if (index !== -1) {
                targetHighlights[index].color = newColor;
                
                // 准备要保存的数据
                const saveData = { highlights };
                
                // 如果是onlyCurrentTab模式，同时更新currentHighlights
                if (settings.onlyCurrentTab) {
                    saveData.currentHighlights = targetHighlights;
                }
                
                await new Promise(resolve => {
                    chrome.storage.local.set(saveData, resolve);
                });
                
                updateHighlightList(targetHighlights);
                
                // 立即更新页面上的高亮颜色
                if (settings.autoHighlight) {
                    await updateHighlights(true); // 强制更新
                }
            }
        } catch (err) {
            console.error('更新高亮颜色时出错:', err);
        }
    }

    // Add editHighlight function
    async function editHighlight(id, newText) {
        if (!isExtensionContextValid()) {
            console.debug('编辑高亮时扩展上下文已失效');
            return;
        }
        
        try {
            const data = await new Promise(resolve => {
                chrome.storage.local.get(['highlights', 'settings'], resolve);
            });
            
            const highlights = data.highlights ?? {};
            highlights.default = highlights.default ?? [];
            highlights.tabs = highlights.tabs ?? {};
            const settings = data.settings ?? {};
            
            let targetHighlights = [];
            let index = -1;
            
            if (settings.onlyCurrentTab) {
                // 编辑当前标签页的高亮词
                const tabs = await new Promise(resolve => {
                    chrome.tabs.query({active: true, currentWindow: true}, resolve);
                });
                
                if (tabs && tabs.length > 0) {
                    const currentTabId = tabs[0].id.toString();
                    highlights.tabs[currentTabId] = highlights.tabs[currentTabId] ?? [];
                    targetHighlights = highlights.tabs[currentTabId];
                    index = targetHighlights.findIndex(h => h.id === id);
                }
            } else {
                // 编辑全局高亮词
                targetHighlights = highlights.default;
                index = targetHighlights.findIndex(h => h.id === id);
            }
            
            if (index !== -1) {
                targetHighlights[index].text = newText;
                
                // 准备要保存的数据
                const saveData = { highlights };
                
                // 如果是onlyCurrentTab模式，同时更新currentHighlights
                if (settings.onlyCurrentTab) {
                    saveData.currentHighlights = targetHighlights;
                }
                
                await new Promise(resolve => {
                    chrome.storage.local.set(saveData, resolve);
                });
                
                updateHighlightList(targetHighlights);
                
                // 立即更新页面上的高亮文本
                if (settings.autoHighlight) {
                    await updateHighlights(true); // 强制更新
                }
            }
        } catch (error) {
            console.error('编辑高亮词时出错:', error);
        }
    }

    // Add deleteHighlight function
    async function deleteHighlight(id) {
        if (!isExtensionContextValid()) {
            console.debug('删除高亮时扩展上下文已失效');
            return;
        }
        
        try {
            const data = await new Promise((resolve) => {
                chrome.storage.local.get(['highlights', 'settings'], resolve);
            });
            
            const highlights = data.highlights ?? {};
            highlights.default = highlights.default ?? [];
            highlights.tabs = highlights.tabs ?? {};
            const settings = data.settings ?? {};
            
            let deletedHighlight = null;
            let targetHighlights = [];
            
            if (settings.onlyCurrentTab) {
                // 从当前标签页删除
                const tabs = await new Promise(resolve => {
                    chrome.tabs.query({active: true, currentWindow: true}, resolve);
                });
                
                if (tabs && tabs.length > 0) {
                    const currentTabId = tabs[0].id.toString();
                    highlights.tabs[currentTabId] = highlights.tabs[currentTabId] ?? [];
                    deletedHighlight = highlights.tabs[currentTabId].find(h => h.id === id);
                    if (deletedHighlight) {
                        highlights.tabs[currentTabId] = highlights.tabs[currentTabId].filter(h => h.id !== id);
                        targetHighlights = highlights.tabs[currentTabId];
                    }
                }
            } else {
                // 从全局删除
                deletedHighlight = highlights.default.find(h => h.id === id);
                if (deletedHighlight) {
                    highlights.default = highlights.default.filter(h => h.id !== id);
                    targetHighlights = highlights.default;
                }
            }
            
            if (!deletedHighlight) return;
            
            // 准备要保存的数据
            const saveData = { highlights };
            
            // 如果是onlyCurrentTab模式，同时更新currentHighlights
            if (settings.onlyCurrentTab) {
                saveData.currentHighlights = targetHighlights;
            }
            
            await new Promise((resolve) => {
                chrome.storage.local.set(saveData, resolve);
            });
            
            updateHighlightList(targetHighlights);
            
            // 发送消息到当前标签页移除高亮
            const tabsResult = await new Promise((resolve) => {
                chrome.tabs.query({active: true, currentWindow: true}, resolve);
            });
            
            if (!tabsResult || !tabsResult.length || !tabsResult[0]) {
                console.debug('没有活动标签页');
                return;
            }
            
            const activeTab = tabsResult[0];
            
            // 检查内容脚本是否准备好
            const isReady = await checkContentScriptReady(activeTab.id);
            if (!isReady) {
                console.debug('内容脚本未准备好，无法删除高亮');
                return;
            }
            
            // 完全重新应用所有高亮，确保状态正确
            const removeResult = await sendMessageSafely(activeTab.id, {
                action: 'removeHighlights',
                force: true
            });
            
            if (!removeResult.success) {
                console.debug('移除高亮失败:', removeResult.error);
                return;
            }
            
            // 确保扩展上下文仍然有效
            if (!isExtensionContextValid()) return;
            
            // 等待一小段时间
            await new Promise(resolve => setTimeout(resolve, 50));
            
            // 重新应用所有高亮
            // 获取要发送的高亮词列表
            let highlightsToSend = [];
            
            if (settings.onlyCurrentTab) {
                // 在onlyCurrentTab模式下，使用当前标签页的高亮词
                const currentTabId = activeTab.id.toString();
                highlightsToSend = highlights.tabs[currentTabId] || [];
            } else {
                // 在全局模式下，使用默认高亮词
                highlightsToSend = highlights.default;
            }
            
            const updateResult = await sendMessageSafely(activeTab.id, {
                action: 'updateHighlights',
                enabled: true,
                force: true,
                highlights: highlightsToSend.map(h => ({
                    ...h,
                    globalMatch: true,
                    processAll: true,
                    multipleMatches: true,
                    maintainAllHighlights: true
                })),
                settings: {
                    ...settings,
                    autoHighlight: true,
                    globalMatch: true,
                    batchSize: 50,
                    processingInterval: 100,
                    processAll: true,
                    multipleMatches: true,
                    maintainAllHighlights: true,
                    reprocessExisting: true
                }
            });
            
            if (!updateResult.success) {
                console.debug('重新应用高亮失败:', updateResult.error);
            }
        } catch (error) {
            console.error('删除高亮时出错:', error);
        }
    }

    // Add style element
    const style = document.createElement('style');
    style.textContent = `
        body {
            width: 400px;
            min-height: 500px;
            margin: 0;
            padding: 0;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, 'Open Sans', 'Helvetica Neue', sans-serif;
            background: #f8f9fa;
            position: relative;
            color: #2c3e50;
            display: flex;
            flex-direction: column;
            overflow: hidden;
        }

        .header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 16px 20px;
            background: linear-gradient(135deg, #1a73e8, #4285f4);
            color: white;
            border-radius: 0;
            margin: 0;
            box-shadow: 0 2px 8px rgba(26, 115, 232, 0.2);
            flex-shrink: 0;
        }

        .header h2 {
            margin: 0;
            font-size: 22px;
            font-weight: 600;
            color: white;
            letter-spacing: 0.3px;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, 'Open Sans', 'Helvetica Neue', sans-serif;
        }

        .content-wrapper {
            flex: 1;
            overflow-y: auto;
            padding: 16px;
            padding-bottom: 60px;
        }

        .main-content {
            display: flex;
            flex-direction: column;
            gap: 16px;
        }

        .settings-group {
            display: flex;
            flex-direction: column;
            gap: 8px;
            margin-bottom: 0;
            padding: 12px;
            background: white;
            border-radius: 8px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.05);
        }

        .settings-row {
            display: flex;
            justify-content: space-between;
            gap: 12px;
        }

        .settings-group label {
            flex: 1;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 6px;
            font-size: 14px;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, 'Open Sans', 'Helvetica Neue', sans-serif;
            color: #4a5568;
            cursor: pointer;
            padding: 8px 12px;
            border-radius: 6px;
            background: #f8fafc;
            border: 1px solid #e2e8f0;
            transition: all 0.2s ease;
            white-space: nowrap;
            min-width: 0;
        }

        .settings-group label:hover {
            background: #f1f5f9;
            border-color: #1a73e8;
            color: #1a73e8;
        }

        .settings-group input[type="checkbox"] {
            width: 14px;
            height: 14px;
            margin: 0;
            cursor: pointer;
            accent-color: #1a73e8;
            flex-shrink: 0;
        }

        .settings-group label:has(input:checked) {
            background: #e8f0fe;
            border-color: #1a73e8;
            color: #1a73e8;
        }

        .input-group {
            display: flex;
            gap: 12px;
            margin-bottom: 0;
            background: white;
            padding: 10px;
            border-radius: 8px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.05);
            align-items: center;
        }

        #highlightWord {
            flex: 1;
            padding: 6px 12px;
            border: 2px solid #e2e8f0;
            border-radius: 8px;
            font-size: 14px;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, 'Open Sans', 'Helvetica Neue', sans-serif;
            transition: all 0.2s ease;
            height: 32px;
        }

        #highlightColor {
            width: 32px;
            height: 32px;
            padding: 3px;
            border: 2px solid #e2e8f0;
            border-radius: 8px;
            cursor: pointer;
            transition: transform 0.2s ease;
        }

        #addWord {
            padding: 6px 16px;
            height: 32px;
            background: #1a73e8;
            color: white;
            border: none;
            border-radius: 8px;
            font-size: 14px;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, 'Open Sans', 'Helvetica Neue', sans-serif;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.2s ease;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        #addWord:hover {
            background: #1557b0;
            transform: translateY(-1px);
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
        }

        #addWord:active {
            transform: translateY(0);
            background: #0d47a1;
            box-shadow: none;
        }

        .search-group {
            position: relative;
            margin-bottom: 0;
            background: white;
            border-radius: 8px;
            padding: 14px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.05);
        }

        #searchHighlight {
            width: 100%;
            padding: 8px 14px 8px 36px;
            border: 2px solid #e2e8f0;
            border-radius: 8px;
            font-size: 14px;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, 'Open Sans', 'Helvetica Neue', sans-serif;
            background: white;
            transition: all 0.2s ease;
            color: #4a5568;
            box-shadow: none;
            box-sizing: border-box;
        }

        #searchHighlight::placeholder {
            color: #a0aec0;
            font-style: italic;
        }

        #searchHighlight:focus {
            outline: none;
            border-color: #1a73e8;
            box-shadow: 0 0 0 3px rgba(26, 115, 232, 0.1);
            background-color: #fff;
        }

        .search-icon {
            position: absolute;
            left: 24px;
            top: 50%;
            transform: translateY(-50%);
            color: #94a3b8;
            pointer-events: none;
            z-index: 1;
        }

        #searchHighlight:focus + .search-icon {
            color: #1a73e8;
        }

        .highlight-container {
            background: white;
            border-radius: 8px;
            padding: 14px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.05);
            overflow-y: auto;
            max-height: 220px;
            min-height: 100px;
            will-change: transform;
            transform: translateZ(0);
            backface-visibility: hidden;
            perspective: 1000px;
        }

        .highlight-container:empty {
            display: flex;
            align-items: center;
            justify-content: center;
            color: #94a3b8;
            font-style: italic;
            font-size: 14px;
        }

        .highlight-container:empty::after {
            content: '暂无高亮词';
        }

        .highlight-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 8px 12px;
            border-radius: 6px;
            background: #f8fafc;
            margin-bottom: 8px;
            transition: all 0.2s ease;
            will-change: transform;
            transform: translateZ(0);
        }

        .word-info {
            display: flex;
            align-items: center;
            gap: 8px;
            flex: 1;
            min-width: 0;
        }

        .text {
            flex: 1;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
            font-size: 13px;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, 'Open Sans', 'Helvetica Neue', sans-serif;
        }

        .color-preview {
            width: 24px;
            height: 24px;
            border-radius: 6px;
            border: 2px solid white;
            box-shadow: 0 0 0 1px #e2e8f0;
            cursor: pointer;
            transition: all 0.2s ease;
        }

        .color-preview:hover {
            transform: scale(1.1);
            box-shadow: 0 0 0 1px #1a73e8;
        }

        .color-preview:active {
            transform: scale(1);
        }

        .match-tag {
            font-size: 12px;
            padding: 3px 8px;
            border-radius: 12px;
            color: white;
            font-weight: 500;
        }

        .match-tag.fuzzy {
            background-color: #34a853;
        }

        .match-tag.whole {
            background-color: #4285f4;
        }

        .bottom-nav {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            display: flex;
            background: white;
            border-top: 1px solid #e2e8f0;
            padding: 8px;
            gap: 8px;
            z-index: 1000;
            box-shadow: 0 -2px 10px rgba(0,0,0,0.05);
        }

        .nav-tab {
            flex: 1;
            padding: 10px;
            text-align: center;
            background: none;
            border: none;
            color: #64748b;
            font-size: 14px;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, 'Open Sans', 'Helvetica Neue', sans-serif;
            font-weight: 500;
            cursor: pointer;
            border-radius: 8px;
            transition: all 0.2s ease;
        }

        .nav-tab:hover {
            background: #f1f5f9;
            color: #1a73e8;
        }

        .nav-tab.active {
            background: #1a73e8;
            color: white;
        }

        .toggle-switch {
            position: relative;
            display: inline-block;
            width: 52px;
            height: 28px;
        }

        .toggle-switch input {
            opacity: 0;
            width: 0;
            height: 0;
        }

        .toggle-slider {
            position: absolute;
            cursor: pointer;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: rgba(255,255,255,0.3);
            transition: .3s;
            border-radius: 34px;
        }

        .toggle-slider:before {
            position: absolute;
            content: "";
            height: 22px;
            width: 22px;
            left: 3px;
            bottom: 3px;
            background-color: white;
            transition: .3s;
            border-radius: 50%;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }

        input:checked + .toggle-slider {
            background-color: rgba(255,255,255,0.8);
        }

        input:checked + .toggle-slider:before {
            transform: translateX(24px);
        }

        .btn {
            padding: 10px 16px;
            border: none;
            border-radius: 8px;
            font-size: 14px;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, 'Open Sans', 'Helvetica Neue', sans-serif;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.2s ease;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
            position: relative;
            overflow: hidden;
        }

        .btn::after {
            content: '';
            position: absolute;
            top: 50%;
            left: 50%;
            width: 5px;
            height: 5px;
            background: rgba(255, 255, 255, 0.5);
            opacity: 0;
            border-radius: 100%;
            transform: scale(1, 1) translate(-50%);
            transform-origin: 50% 50%;
        }

        .btn:active::after {
            animation: ripple 0.6s ease-out;
        }

        @keyframes ripple {
            0% {
                transform: scale(0, 0);
                opacity: 0.5;
            }
            100% {
                transform: scale(40, 40);
                opacity: 0;
            }
        }

        .btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.15);
        }

        .btn:active {
            transform: translateY(0);
            box-shadow: 0 1px 2px rgba(0, 0, 0, 0.1);
        }

        .btn-gray {
            background: #e2e8f0;
            color: #4a5568;
            transition: all 0.3s ease;
        }

        .btn-gray:hover:not(:disabled) {
            background: #cbd5e1;
            color: #2d3748;
        }

        .btn-gray:active:not(:disabled) {
            background: #a0aec0;
        }

        .btn-gray:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            background: #e2e8f0;
            color: #a0aec0;
            transform: none;
            box-shadow: none;
        }

        .btn-gray.has-selected {
            background: #ef4444;
            color: white;
        }

        .btn-gray.has-selected:hover {
            background: #dc2626;
        }

        .btn-gray.has-selected:active {
            background: #b91c1c;
        }

        .btn-danger {
            background: #ef4444;
            color: white;
        }

        .btn-danger:hover {
            background: #dc2626;
        }

        .btn-danger:active {
            background: #b91c1c;
        }

        .btn-primary {
            background: #3b82f6;
            color: white;
        }

        .btn-primary:hover {
            background: #2563eb;
        }

        .btn-primary:active {
            background: #1d4ed8;
        }

        .btn-primary.importing {
            cursor: wait;
            opacity: 0.8;
            transform: none;
            box-shadow: none;
        }

        .btn-secondary {
            background: #4b5563;
            color: white;
        }

        .btn-secondary:hover {
            background: #374151;
        }

        .btn-secondary:active {
            background: #1f2937;
        }

        .control-btn {
            width: 32px;
            height: 32px;
            border: 1px solid #e2e8f0;
            border-radius: 6px;
            background: white;
            color: #64748b;
            font-size: 16px;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.2s ease;
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
        }

        .control-btn:hover {
            border-color: #1a73e8;
            color: #1a73e8;
            transform: translateY(-1px);
            box-shadow: 0 3px 6px rgba(0, 0, 0, 0.15);
        }

        .control-btn:active {
            transform: translateY(0);
            box-shadow: 0 1px 2px rgba(0, 0, 0, 0.1);
        }

        .delete-btn:hover {
            border-color: #ef4444;
            color: #ef4444;
        }

        .delete-btn:active {
            background: #fee2e2;
        }

        input[type="checkbox"] {
            position: relative;
            width: 16px;
            height: 16px;
            margin: 0;
            cursor: pointer;
            border: 2px solid #e2e8f0;
            border-radius: 4px;
            background: white;
            transition: all 0.2s ease;
            appearance: none;
            -webkit-appearance: none;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        input[type="checkbox"]:hover {
            border-color: #1a73e8;
        }

        input[type="checkbox"]:checked {
            background: #1a73e8;
            border-color: #1a73e8;
        }

        input[type="checkbox"]:checked::after {
            content: '';
            position: absolute;
            width: 4px;
            height: 8px;
            border: solid white;
            border-width: 0 2px 2px 0;
            transform: rotate(45deg) translate(-1px, -1px);
        }

        input[type="checkbox"]:active {
            transform: scale(0.9);
        }

        .settings-group label:has(input[type="checkbox"]:checked) {
            background: #e8f0fe;
            border-color: #1a73e8;
            color: #1a73e8;
        }

        .settings-group label:has(input[type="checkbox"]:hover) {
            border-color: #1a73e8;
        }

        #highlightWord:focus {
            border-color: #1a73e8;
            box-shadow: 0 0 0 3px rgba(26, 115, 232, 0.1);
        }

        #highlightColor:hover {
            transform: scale(1.1);
            border-color: #1a73e8;
        }

        #highlightColor:active {
            transform: scale(1);
        }

        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0,0,0,0.5);
            z-index: 2000;
            backdrop-filter: blur(4px);
        }

        .modal-content {
            position: fixed;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: white;
            padding: 24px;
            border-radius: 16px;
            width: 90%;
            max-height: 80vh;
            overflow-y: auto;
            box-shadow: 0 4px 20px rgba(0,0,0,0.15);
        }

        .modal-close {
            position: absolute;
            top: 16px;
            right: 16px;
            border: none;
            background: none;
            font-size: 24px;
            cursor: pointer;
            color: #64748b;
            transition: color 0.2s ease;
        }

        .modal-close:hover {
            color: #1a73e8;
        }

        .about-content h3,
        .privacy-content h3,
        .permission-content h3 {
            color: #1a73e8;
            margin-bottom: 20px;
            font-size: 20px;
        }

        .about-item,
        .permission-item,
        .data-item {
            margin-bottom: 28px;
            padding: 20px;
            background: #f8fafc;
            border-radius: 12px;
            transition: all 0.2s ease;
        }

        .about-item:hover,
        .permission-item:hover,
        .data-item:hover {
            background: white;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
        }

        .about-item h4,
        .permission-item h5,
        .data-item h5 {
            color: #1a73e8;
            margin-bottom: 12px;
            font-size: 16px;
        }

        .about-item p,
        .permission-item p,
        .data-item p {
            margin: 8px 0;
            color: #4a5568;
            line-height: 1.6;
        }

        .permission-item ul,
        .data-item ul {
            margin: 12px 0;
            padding-left: 20px;
        }

        .permission-item li,
        .data-item li {
            margin-bottom: 8px;
            color: #4a5568;
            line-height: 1.6;
        }

        ::-webkit-scrollbar {
            width: 8px;
        }

        ::-webkit-scrollbar-track {
            background: #f1f5f9;
            border-radius: 4px;
        }

        ::-webkit-scrollbar-thumb {
            background: #cbd5e1;
            border-radius: 4px;
        }

        ::-webkit-scrollbar-thumb:hover {
            background: #94a3b8;
        }

        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }

        .tab-pane {
            display: none;
            opacity: 0;
            transition: opacity 0.3s ease;
            height: 100%;
        }

        .tab-pane.active {
            display: block;
            opacity: 1;
        }

        .settings-content,
        .about-content {
            height: 100%;
            overflow-y: auto;
            padding: 0;
        }

        .settings-section,
        .about-section {
            background: white;
            border-radius: 12px;
            padding: 24px;
            margin-bottom: 20px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
        }

        .settings-section:last-child,
        .about-section:last-child {
            margin-bottom: 0;
        }

        .import-group {
            margin-bottom: 0;
            display: flex;
            gap: 8px;
            padding: 14px;
            background: white;
            border-radius: 8px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.05);
        }

        .import-group .btn {
            padding: 8px 12px;
            border: none;
            border-radius: 6px;
            font-size: 13px;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, 'Open Sans', 'Helvetica Neue', sans-serif;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.15s ease;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            text-align: center;
            flex: 1;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
            min-width: 80px;
            position: relative;
            transform: translateY(0);
        }

        .import-group .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0,0,0,0.15);
            filter: brightness(110%);
        }

        .import-group .btn:active {
            transform: translateY(0);
            box-shadow: 0 1px 2px rgba(0,0,0,0.1);
            filter: brightness(90%);
        }

        .import-group .btn::after {
            content: '';
            position: absolute;
            top: 50%;
            left: 50%;
            width: 100%;
            height: 100%;
            background: radial-gradient(circle, rgba(255,255,255,0.3) 0%, transparent 70%);
            transform: translate(-50%, -50%) scale(0);
            opacity: 0;
            transition: transform 0.3s ease, opacity 0.3s ease;
            pointer-events: none;
        }

        .import-group .btn:active::after {
            transform: translate(-50%, -50%) scale(2);
            opacity: 1;
            transition: 0s;
        }

        .import-group .btn-gray {
            background: #e2e8f0;
            color: #4a5568;
        }

        .import-group .btn-gray:disabled {
            opacity: 0.5;
            cursor: not-allowed;
            transform: none !important;
            box-shadow: none !important;
            filter: none !important;
        }

        .import-group .btn-gray:not(:disabled):hover {
            background: #cbd5e1;
            color: #1a202c;
        }

        .import-group .btn-gray:not(:disabled):active {
            background: #94a3b8;
        }

        .import-group .btn-danger {
            background: #ef4444;
            color: white;
        }

        .import-group .btn-danger:hover {
            background: #dc2626;
        }

        .import-group .btn-danger:active {
            background: #b91c1c;
        }

        .import-group .btn-primary {
            background: #3b82f6;
            color: white;
            position: relative;
            overflow: hidden;
        }

        .import-group .btn-primary:hover {
            background: #2563eb;
        }

        .import-group .btn-primary:active {
            background: #1d4ed8;
        }

        .import-group .btn-primary.importing {
            cursor: wait;
            pointer-events: none;
        }

        .import-group .btn-secondary {
            background: #4b5563;
            color: white;
        }

        .import-group .btn-secondary:hover {
            background: #374151;
        }

        .import-group .btn-secondary:active {
            background: #1f2937;
        }

        .import-group .btn-primary .progress-overlay {
            position: absolute;
            left: 0;
            top: 0;
            height: 100%;
            background: rgba(255, 255, 255, 0.2);
            transition: width 0.3s ease;
            pointer-events: none;
        }

        .import-group .btn-primary .button-text {
            position: relative;
            z-index: 1;
        }

        .controls {
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .highlight-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 12px;
            padding: 0 6px;
        }

        .highlight-list-title {
            font-size: 16px;
            font-weight: 600;
            color: #1a73e8;
            margin-bottom: 0;
            padding: 0;
        }

        .select-all-label {
            display: flex;
            align-items: center;
            gap: 8px;
            font-size: 14px;
            color: #4a5568;
            cursor: pointer;
            user-select: none;
            position: relative;
            padding-left: 28px;
        }

        .select-all-label input[type="checkbox"] {
            position: absolute;
            opacity: 0;
            cursor: pointer;
            height: 0;
            width: 0;
        }

        .checkmark {
            position: absolute;
            left: 0;
            height: 20px;
            width: 20px;
            background-color: #fff;
            border: 2px solid #e2e8f0;
            border-radius: 4px;
            transition: all 0.2s ease;
        }

        .select-all-label:hover .checkmark {
            border-color: #1a73e8;
        }

        .select-all-label input:checked ~ .checkmark {
            background-color: #1a73e8;
            border-color: #1a73e8;
        }

        .checkmark:after {
            content: "";
            position: absolute;
            display: none;
            left: 6px;
            top: 2px;
            width: 5px;
            height: 10px;
            border: solid white;
            border-width: 0 2px 2px 0;
            transform: rotate(45deg);
        }

        .select-all-label input:checked ~ .checkmark:after {
            display: block;
        }

        .label-text {
            font-size: 14px;
            color: #4a5568;
            transition: color 0.2s ease;
        }

        .select-all-label:hover .label-text {
            color: #1a73e8;
        }

        .update-log {
            margin-top: 12px;
        }

        .update-log p {
            margin: 8px 0;
            color: #1a73e8;
            font-weight: 500;
        }

        .update-log ul {
            margin: 8px 0;
            padding-left: 20px;
            list-style-type: disc;
        }

        .update-log li {
            margin: 6px 0;
            color: #4a5568;
            line-height: 1.5;
        }

        @media (max-width: 400px) {
            .settings-group label {
                font-size: 12px;
                padding: 4px 6px;
            }
        }
    `;
    document.head.appendChild(style);

    // Add progress bar styles
    const progressStyles = document.createElement('style');
    progressStyles.textContent = `
        .import-group .btn-primary {
            background: #3b82f6;
            color: white;
            position: relative;
            overflow: hidden;
        }

        .import-group .btn-primary:hover {
            background: #2563eb;
        }

        .import-group .btn-primary .progress-overlay {
            position: absolute;
            left: 0;
            top: 0;
            height: 100%;
            background: rgba(255, 255, 255, 0.2);
            transition: width 0.3s ease;
            pointer-events: none;
        }

        .import-group .btn-primary .button-text {
            position: relative;
            z-index: 1;
        }

        .import-group .btn-primary.importing {
            cursor: wait;
            pointer-events: none;
        }
    `;
    document.head.appendChild(progressStyles);

    // Initialize state and settings
    chrome.storage.local.get(['enabled', 'highlights', 'settings'], function(data) {
        const settings = data.settings || {};
        
        // 设置默认值
        const defaultSettings = {
            autoHighlight: true,  // 默认开启自动高亮
            caseSensitive: false,
            globalMatch: true,
            highlightDelay: 800,
            renderMode: 'stable',
            useHardwareAcceleration: true,
            usePositionLocking: true,
            ...settings
        };
        
        // 更新复选框状态
        enableHighlight.checked = data.enabled !== undefined ? data.enabled : true;
        autoHighlight.checked = defaultSettings.autoHighlight;
        caseSensitive.checked = defaultSettings.caseSensitive;
        globalMatch.checked = defaultSettings.globalMatch;
        onlyCurrentTab.checked = defaultSettings.onlyCurrentTab;
        
        // 保存默认设置到local存储
        chrome.storage.local.set({ settings: defaultSettings }, function() {
            // 根据onlyCurrentTab设置加载对应的高亮词
            loadHighlightsByMode(data, defaultSettings.onlyCurrentTab);
        });
    });
    
    // 根据模式加载高亮词的函数
    function loadHighlightsByMode(data, onlyCurrentTab) {
        if (onlyCurrentTab) {
            // 获取当前标签页ID并加载该标签页的高亮词
            chrome.tabs.query({active: true, currentWindow: true}, function(tabs) {
                if (tabs && tabs.length > 0) {
                    const currentTabId = tabs[0].id.toString();
                    
                    // 确保highlights结构存在
                    const highlights = data.highlights ?? {};
                    highlights.tabs = highlights.tabs ?? {};
                    
                    // 获取当前标签页的高亮词
                    let tabHighlights = highlights.tabs[currentTabId] ?? [];
                    
                    // 同时检查currentHighlights，确保数据一致性
                    chrome.storage.local.get(['currentHighlights'], function(result) {
                        const currentHighlights = result.currentHighlights ?? [];
                        
                        // 如果currentHighlights有数据但tabs中没有，使用currentHighlights
                        if (currentHighlights.length > 0 && tabHighlights.length === 0) {
                            tabHighlights = currentHighlights;
                            highlights.tabs[currentTabId] = tabHighlights;
                            chrome.storage.local.set({ highlights }, function() {
                                console.debug('已同步currentHighlights到tabs存储:', tabHighlights.length);
                            });
                        } 
                        // 如果tabs中有数据但currentHighlights没有或不一致，更新currentHighlights
                        else if (tabHighlights.length > 0 && JSON.stringify(tabHighlights) !== JSON.stringify(currentHighlights)) {
                            chrome.storage.local.set({ currentHighlights: tabHighlights }, function() {
                                console.debug('已同步tabs存储到currentHighlights:', tabHighlights.length);
                            });
                        }
                        
                        console.debug('onlyCurrentTab模式：加载标签页高亮词:', tabHighlights.length);
                        updateHighlightList(tabHighlights);
                    });
                } else {
                    updateHighlightList([]);
                }
            });
        } else {
            // 加载全局高亮词
            const globalHighlights = data.highlights?.default ?? [];
            console.debug('全局模式：加载全局高亮词:', globalHighlights.length);
            updateHighlightList(globalHighlights);
        }
    }

    // Add enable/disable toggle handler with performance optimization
    enableHighlight.addEventListener('change', function() {
        const enabled = this.checked;
        chrome.storage.local.set({ enabled: enabled }, function() {
            chrome.tabs.query({active: true, currentWindow: true}, function(tabs) {
                if (tabs[0]) {
                    chrome.storage.local.get(['highlights', 'settings'], function(data) {
                        const settings = data.settings || {};
                        let highlights = [];
                        
                        // 根据onlyCurrentTab设置选择正确的高亮词来源
                        if (settings.onlyCurrentTab) {
                            const currentTabId = tabs[0].id.toString();
                            highlights = data.highlights?.tabs?.[currentTabId] ?? [];
                        } else {
                            highlights = data.highlights?.default ?? [];
                        }
                        
                        chrome.tabs.sendMessage(tabs[0].id, {
                            action: enabled ? 'enableHighlight' : 'disableHighlight',
                            force: true,
                            maintain: true,
                            highlights: highlights,
                            settings: {
                                ...settings,
                                batchSize: 50,
                                processingInterval: 100
                            }
                        }, function(response) {
                            if (chrome.runtime.lastError) {
                                console.error('切换高亮状态时发生错误:', chrome.runtime.lastError);
                                return;
                            }
                        });
                    });
                }
            });
        });
    });

    // Add window focus handler to maintain highlights
    window.addEventListener('focus', function() {
        chrome.tabs.query({active: true, currentWindow: true}, function(tabs) {
            if (tabs[0]) {
                chrome.tabs.sendMessage(tabs[0].id, {
                    action: 'maintainHighlight',
                    maintain: true
                });
            }
        });
    });

    // Add window blur handler to maintain highlights
    window.addEventListener('blur', function() {
        chrome.tabs.query({active: true, currentWindow: true}, function(tabs) {
            if (tabs[0]) {
                chrome.tabs.sendMessage(tabs[0].id, {
                    action: 'maintainHighlight',
                    maintain: true
                });
            }
        });
    });

    // Reset delete button state
    function resetDeleteButtonState() {
        // Reset select all checkbox
        selectAll.checked = false;
        // Reset delete button state
        deleteSelected.classList.remove('has-selected');
        deleteSelected.style.transition = 'all 0.3s ease';
        deleteSelected.style.background = '#e2e8f0';
        deleteSelected.style.color = '#4a5568';
        deleteSelected.disabled = true;
    }

    // Update delete button state
    function updateDeleteButtonState() {
        const checkedBoxes = highlightContainer.querySelectorAll('.checkbox:checked');
        const hasChecked = checkedBoxes.length > 0;
        const totalBoxes = highlightContainer.querySelectorAll('.checkbox').length;
        
        // 更新已选中数量显示
        const searchTerm = document.getElementById('searchHighlight')?.value?.toLowerCase().trim() || '';
        chrome.storage.local.get(['highlights', 'settings', 'currentHighlights'], function(data) {
            const settings = data.settings || {};
            let highlights = [];
            
            if (settings.onlyCurrentTab) {
                // 在onlyCurrentTab模式下，使用currentHighlights或当前标签页的高亮词
                chrome.tabs.query({active: true, currentWindow: true}, function(tabs) {
                    if (tabs && tabs.length > 0) {
                        const currentTabId = tabs[0].id.toString();
                        // 首先尝试从tabs中获取高亮词
                        const tabHighlights = data.highlights?.tabs?.[currentTabId] ?? [];
                        
                        // 如果tabs中没有，则使用currentHighlights
                        if (tabHighlights.length === 0) {
                            highlights = data.currentHighlights ?? [];
                        } else {
                            highlights = tabHighlights;
                        }
                        
                        const displayHighlights = searchTerm
                            ? highlights.filter(h => h.text.toLowerCase().includes(searchTerm))
                            : highlights;
                        
                        updateHighlightCountDisplay(highlights.length, displayHighlights.length, checkedBoxes.length);
                    } else {
                        updateHighlightCountDisplay(0, 0, checkedBoxes.length);
                    }
                });
            } else {
                // 在全局模式下，使用default高亮词
                highlights = data.highlights?.default ?? [];
                const displayHighlights = searchTerm
                    ? highlights.filter(h => h.text.toLowerCase().includes(searchTerm))
                    : highlights;
                
                updateHighlightCountDisplay(highlights.length, displayHighlights.length, checkedBoxes.length);
            }
        });
        
        // Update delete button state
        deleteSelected.disabled = !hasChecked;
        
        if (hasChecked) {
            deleteSelected.classList.add('has-selected');
            deleteSelected.style.transition = 'all 0.3s ease';
            deleteSelected.style.background = '#ef4444';
            deleteSelected.style.color = 'white';
        } else {
            deleteSelected.classList.remove('has-selected');
            deleteSelected.style.transition = 'all 0.3s ease';
            deleteSelected.style.background = '#e2e8f0';
            deleteSelected.style.color = '#4a5568';
        }
        
        // Update select all checkbox state
        selectAll.checked = totalBoxes > 0 && checkedBoxes.length === totalBoxes;
    }

    // Add event listeners for delete selected button
    deleteSelected.addEventListener('mouseover', function() {
        if (!this.disabled && this.classList.contains('has-selected')) {
            this.style.background = '#dc2626';
        }
    });

    deleteSelected.addEventListener('mouseout', function() {
        if (!this.disabled && this.classList.contains('has-selected')) {
            this.style.background = '#ef4444';
        }
    });

    deleteSelected.addEventListener('mousedown', function() {
        if (!this.disabled && this.classList.contains('has-selected')) {
            this.style.background = '#b91c1c';
        }
    });

    deleteSelected.addEventListener('mouseup', function() {
        if (!this.disabled && this.classList.contains('has-selected')) {
            this.style.background = '#ef4444';
        }
    });

    // Select All functionality
    selectAll.addEventListener('change', function() {
        const checkboxes = highlightContainer.querySelectorAll('.checkbox');
        checkboxes.forEach(checkbox => checkbox.checked = this.checked);
        updateDeleteButtonState();
        
        // 直接更新勾选数量显示
        const checkedCount = this.checked ? checkboxes.length : 0;
        const searchTerm = document.getElementById('searchHighlight')?.value?.toLowerCase().trim() || '';
        chrome.storage.local.get(['highlights', 'settings', 'currentHighlights'], function(data) {
            const settings = data.settings || {};
            let highlights = [];
            
            if (settings.onlyCurrentTab) {
                // 在onlyCurrentTab模式下，使用currentHighlights或当前标签页的高亮词
                chrome.tabs.query({active: true, currentWindow: true}, function(tabs) {
                    if (tabs && tabs.length > 0) {
                        const currentTabId = tabs[0].id.toString();
                        // 首先尝试从tabs中获取高亮词
                        const tabHighlights = data.highlights?.tabs?.[currentTabId] ?? [];
                        
                        // 如果tabs中没有，则使用currentHighlights
                        if (tabHighlights.length === 0) {
                            highlights = data.currentHighlights ?? [];
                        } else {
                            highlights = tabHighlights;
                        }
                        
                        const displayHighlights = searchTerm
                            ? highlights.filter(h => h.text.toLowerCase().includes(searchTerm))
                            : highlights;
                        
                        updateHighlightCountDisplay(highlights.length, displayHighlights.length, checkedCount);
                    } else {
                        updateHighlightCountDisplay(0, 0, checkedCount);
                    }
                });
            } else {
                // 在全局模式下，使用default高亮词
                highlights = data.highlights?.default ?? [];
                const displayHighlights = searchTerm
                    ? highlights.filter(h => h.text.toLowerCase().includes(searchTerm))
                    : highlights;
                
                updateHighlightCountDisplay(highlights.length, displayHighlights.length, checkedCount);
            }
        });
    });

    // Delete selected highlights
    deleteSelected.addEventListener('click', function() {
        const checkedItems = highlightContainer.querySelectorAll('.checkbox:checked');
        const itemIds = Array.from(checkedItems).map(checkbox => 
            checkbox.closest('.highlight-item').dataset.id
        );
        
        chrome.storage.local.get(['highlights', 'settings'], function(data) {
            const highlights = data.highlights ?? {};
            highlights.default = highlights.default ?? [];
            highlights.tabs = highlights.tabs ?? {};
            const settings = data.settings ?? {};
            
            let targetHighlights = [];
            
            if (settings.onlyCurrentTab) {
                // 从当前标签页删除选中的高亮词
                chrome.tabs.query({active: true, currentWindow: true}, function(tabs) {
                    if (tabs && tabs.length > 0) {
                        const currentTabId = tabs[0].id.toString();
                        highlights.tabs[currentTabId] = highlights.tabs[currentTabId] ?? [];
                        highlights.tabs[currentTabId] = highlights.tabs[currentTabId].filter(h => !itemIds.includes(h.id));
                        targetHighlights = highlights.tabs[currentTabId];
                        
                        chrome.storage.local.set({ highlights }, function() {
                            updateHighlightList(targetHighlights);
                            resetDeleteButtonState();
                            
                            // 在onlyCurrentTab模式下，同步更新currentHighlights
                            chrome.storage.local.set({
                                currentHighlights: targetHighlights
                            }, function() {
                                console.debug('删除操作：已更新currentHighlights');
                                // 调用popup.js自己的updateHighlights函数
                                updateHighlights(true);
                            });
                        });
                    }
                });
            } else {
                // 从全局删除选中的高亮词
                highlights.default = highlights.default.filter(h => !itemIds.includes(h.id));
                targetHighlights = highlights.default;
            
                chrome.storage.local.set({ highlights }, function() {
                    updateHighlightList(targetHighlights);
                    resetDeleteButtonState();
                    
                    // 调用popup.js自己的updateHighlights函数
                    updateHighlights(true);
                });
            }
        });
    });

    // Clear all highlights
    clearAll.addEventListener('click', function() {
        if (confirm('确定要清除所有高亮词吗？')) {
            chrome.storage.local.get(['highlights', 'settings'], function(data) {
                const highlights = data.highlights ?? {};
                highlights.default = highlights.default ?? [];
                highlights.tabs = highlights.tabs ?? {};
                const settings = data.settings ?? {};
                
                if (settings.onlyCurrentTab) {
                    // 清除当前标签页的高亮词
                    chrome.tabs.query({active: true, currentWindow: true}, function(tabs) {
                        if (tabs && tabs.length > 0) {
                            const currentTabId = tabs[0].id.toString();
                            highlights.tabs[currentTabId] = [];
                            
                            chrome.storage.local.set({ highlights }, function() {
                                updateHighlightList([]);
                                updateHighlightCountDisplay(0, 0, 0);
                                resetDeleteButtonState();
                                
                                // 在onlyCurrentTab模式下，同步清空currentHighlights
                                chrome.storage.local.set({
                                    currentHighlights: []
                                }, function() {
                                    console.debug('清除全部操作：已清空currentHighlights');
                                    // 调用popup.js自己的updateHighlights函数
                                    updateHighlights(true);
                                });
                            });
                        }
                    });
                } else {
                    // 清除全局高亮词
                    highlights.default = [];
                    
                    chrome.storage.local.set({ highlights }, function() {
                        updateHighlightList([]);
                        updateHighlightCountDisplay(0, 0, 0);
                        resetDeleteButtonState();
                        
                        // 调用popup.js自己的updateHighlights函数
                        updateHighlights(true);
                    });
                }
            });
        }
    });

    // Set match type change handler
    async function handleMatchTypeChange(setting, value) {
        try {
            const data = await new Promise(resolve => {
                chrome.storage.local.get(['settings', 'highlights'], resolve);
            });
            
            const settings = data.settings ?? {};
            settings[setting] = value;
            
            await new Promise(resolve => {
                chrome.storage.local.set({ settings }, resolve);
            });
            
            // 如果是onlyCurrentTab设置改变，重新加载高亮词列表
            if (setting === 'onlyCurrentTab') {
                // 获取当前标签页信息
                const tabs = await new Promise(resolve => {
                    chrome.tabs.query({active: true, currentWindow: true}, resolve);
                });
                
                if (tabs && tabs[0]) {
                    const currentTabId = tabs[0].id.toString();
                    const highlights = data.highlights ?? {};
                    highlights.tabs = highlights.tabs ?? {};
                    
                    if (value) {
                        // 切换到onlyCurrentTab模式
                        // 确保当前标签页有独立的高亮词存储
                        const tabHighlights = highlights.tabs[currentTabId] ?? [];
                        
                        // 更新currentHighlights为当前标签页的高亮词
                        await new Promise(resolve => {
                            chrome.storage.local.set({ currentHighlights: tabHighlights }, resolve);
                        });
                        
                        console.debug('切换到onlyCurrentTab模式，当前标签页高亮词:', tabHighlights.length);
                    } else {
                        // 切换到全局模式
                        // 清除currentHighlights，使用全局高亮词
                        await new Promise(resolve => {
                            chrome.storage.local.remove('currentHighlights', resolve);
                        });
                        
                        console.debug('切换到全局模式');
                    }
                }
                
                // 使用更新后的数据重新加载
                const updatedData = { ...data, settings };
                loadHighlightsByMode(updatedData, value);
                
                // 立即更新页面高亮
                await updateHighlights(true);
            }
            
            // Send specific setting update message
            const tabs = await new Promise(resolve => {
                chrome.tabs.query({active: true, currentWindow: true}, resolve);
            });
            
            if (tabs && tabs[0]) {
                chrome.tabs.sendMessage(tabs[0].id, {
                    action: 'updateSetting',
                    setting: setting,
                    value: value,
                    maintain: true
                });
            }
        } catch (error) {
            console.error('处理设置变更时出错:', error);
        }
    }

    caseSensitive.addEventListener('change', function() {
        handleMatchTypeChange('caseSensitive', this.checked);
    });

    globalMatch.addEventListener('change', function() {
        handleMatchTypeChange('globalMatch', this.checked);
    });

    onlyCurrentTab.addEventListener('change', function() {
        handleMatchTypeChange('onlyCurrentTab', this.checked);
    });

    // Add highlight word
    addWordBtn.addEventListener('click', async function() {
        if (!isExtensionContextValid()) {
            console.debug('添加高亮词时扩展上下文已失效');
            return;
        }
        
        const word = highlightWord.value.trim();
        if (!word) return;

        try {
            const data = await new Promise(resolve => {
                chrome.storage.local.get(['highlights', 'settings'], resolve);
            });
            
            const highlights = data.highlights ?? {};
            highlights.default = highlights.default ?? [];
            highlights.tabs = highlights.tabs ?? {};
            const settings = data.settings ?? {};

            let targetHighlights = [];
            let currentTabId = null;
            
            if (settings.onlyCurrentTab) {
                // 获取当前标签页ID
                const tabsResult = await new Promise(resolve => {
                    chrome.tabs.query({active: true, currentWindow: true}, resolve);
                });
                
                if (!tabsResult || !tabsResult.length || !tabsResult[0]) {
                    console.debug('没有活动标签页');
                    return;
                }
                
                currentTabId = tabsResult[0].id.toString();
                highlights.tabs[currentTabId] = highlights.tabs[currentTabId] ?? [];
                targetHighlights = highlights.tabs[currentTabId];
            } else {
                targetHighlights = highlights.default;
            }

            // 修改高亮词总数限制为100000
            if (targetHighlights.length >= 100000) {
                alert('高亮词总数已达到上限（100000条），无法添加更多！');
                return;
            }

            // 检查是否已存在相同的高亮词（不区分大小写）
            const existingHighlight = targetHighlights.find(h => 
                h.text.toLowerCase() === word.toLowerCase()
            );

            if (!existingHighlight) {
                const newHighlight = {
                    text: word,
                    color: highlightColor.value,
                    id: Date.now() + '-' + Math.random().toString(36).substr(2, 9),
                    caseSensitive: settings.caseSensitive,
                    globalMatch: true,
                    processAll: true,
                    multipleMatches: true,
                    maintainAllHighlights: true
                };
                
                targetHighlights.push(newHighlight);

                // 准备要保存的数据
                const saveData = { highlights };
                
                // 如果是onlyCurrentTab模式，同时更新currentHighlights
                if (settings.onlyCurrentTab) {
                    saveData.currentHighlights = targetHighlights;
                }

                await new Promise(resolve => {
                    chrome.storage.local.set(saveData, resolve);
                });
                
                highlightWord.value = '';
                updateHighlightList(targetHighlights);
                
                // 直接发送消息到当前标签页应用新的高亮词
                let activeTab;
                if (settings.onlyCurrentTab && currentTabId) {
                    // 如果是标签页模式，使用之前获取的标签页信息
                    const tabsResult = await new Promise(resolve => {
                        chrome.tabs.query({active: true, currentWindow: true}, resolve);
                    });
                    activeTab = tabsResult[0];
                } else {
                    // 如果是全局模式，重新获取当前标签页
                    const tabsResult = await new Promise(resolve => {
                        chrome.tabs.query({active: true, currentWindow: true}, resolve);
                    });
                    
                    if (!tabsResult || !tabsResult.length || !tabsResult[0]) {
                        console.debug('没有活动标签页');
                        return;
                    }
                    
                    activeTab = tabsResult[0];
                }
                
                // 检查内容脚本是否准备好
                const isReady = await checkContentScriptReady(activeTab.id);
                if (!isReady) {
                    console.debug('内容脚本未准备好，无法应用高亮');
                    return;
                }
                
                // 完全重新应用所有高亮，确保状态正确
                const removeResult = await sendMessageSafely(activeTab.id, {
                    action: 'removeHighlights',
                    force: true
                });
                
                if (!removeResult.success) {
                    console.debug('移除高亮失败:', removeResult.error);
                }
                
                // 确保扩展上下文仍然有效
                if (!isExtensionContextValid()) return;
                
                // 等待一小段时间
                await new Promise(resolve => setTimeout(resolve, 50));
                
                // 获取要发送的高亮词列表
                let highlightsToSend = targetHighlights;
                
                // 数据已在保存时同步更新，无需重复处理
                
                // 重新应用所有高亮
                const updateResult = await sendMessageSafely(activeTab.id, {
                    action: 'updateHighlights',
                    enabled: true,
                    force: true,
                    highlights: highlightsToSend.map(h => ({
                        ...h,
                        globalMatch: true,
                        processAll: true,
                        multipleMatches: true,
                        maintainAllHighlights: true
                    })),
                    settings: {
                        ...settings,
                        autoHighlight: true,
                        globalMatch: true,
                        batchSize: 50,
                        processingInterval: 100,
                        processAll: true,
                        multipleMatches: true,
                        maintainAllHighlights: true,
                        reprocessExisting: true
                    }
                });
                
                if (!updateResult.success) {
                    console.debug('应用高亮失败:', updateResult.error);
                }
            } else {
                alert('该高亮词已存在！');
                highlightWord.value = '';
            }
        } catch (error) {
            console.error('添加高亮词时出错:', error);
        }
    });

    // Import/Export functionality
    importBtn.addEventListener('click', function() {
        fileInput.value = '';
        fileInput.click();
    });

    fileInput.addEventListener('change', function() {
        if (this.files.length === 0) return;
        handleImport(this.files[0]);
    });

    async function handleImport(file) {
        try {
            const words = await parseFile(file);
            if (words && words.length > 0) {
                if (confirm(`找到 ${words.length} 个词语，是否全部导入？`)) {
                    importBtn.classList.add('importing');
                    await importWords(words);
                    alert('导入完成！');
                }
            } else {
                alert('未在文件中找到有效的词语！');
            }
        } catch (error) {
            alert('导入失败：' + error.message);
        } finally {
            importBtn.classList.remove('importing');
            fileInput.value = '';
        }
    }

    async function parseFile(file) {
        if (file.name.toLowerCase().endsWith('.txt')) {
            const text = await readFileAsText(file);
            const words = new Set();
            const lines = text.split(/[\n\r]+/);
            
            for (const line of lines) {
                const lineWords = line.split(/[,;\t\s]+/)
                    .map(word => word.trim())
                    .filter(word => word && word.length > 0);
                lineWords.forEach(word => words.add(word));
            }
            
            return Array.from(words);
        } else {
            throw new Error('目前仅支持 .txt 文本文件导入！');
        }
    }

    function readFileAsText(file) {
        return new Promise((resolve, reject) => {
            const reader = new FileReader();
            reader.onload = e => resolve(e.target.result);
            reader.onerror = () => reject(new Error('文件读取失败！'));
            reader.readAsText(file, 'UTF-8');
        });
    }

    async function importWords(words) {
        try {
            if (!Array.isArray(words) || words.length === 0) {
                alert('没有检测到有效的高亮词！');
                return;
            }
            
            // 获取当前的highlights数据
            const data = await new Promise((resolve, reject) => {
                chrome.storage.local.get(['highlights', 'settings'], (result) => {
                    if (chrome.runtime.lastError) {
                        reject(chrome.runtime.lastError);
                    } else {
                        resolve(result);
                    }
                });
            });
            
            const highlights = data.highlights ?? { default: [] };
            const settings = data.settings ?? {};

            // 根据onlyCurrentTab设置选择正确的高亮词来源
            let targetHighlights = [];
            let currentTabId = null;
            if (settings.onlyCurrentTab) {
                // 获取当前标签页ID
                const tabs = await new Promise(resolve => {
                    chrome.tabs.query({active: true, currentWindow: true}, resolve);
                });
                if (tabs && tabs.length > 0) {
                    currentTabId = tabs[0].id.toString();
                    highlights.tabs = highlights.tabs ?? {};
                    highlights.tabs[currentTabId] = highlights.tabs[currentTabId] ?? [];
                    targetHighlights = highlights.tabs[currentTabId];
                } else {
                    alert('无法获取当前标签页信息');
                    return;
                }
            } else {
                highlights.default = highlights.default ?? [];
                targetHighlights = highlights.default;
            }
            
            // 检查是否超过100000个词的限制
            if (targetHighlights.length + words.length > 100000) {
                alert(`导入失败：当前已有${targetHighlights.length}个高亮词，新增${words.length}个词将超过100000个限制！`);
                return;
            }

            let addedCount = 0;
            const batchSize = 50; // 减小批量处理数量，提高稳定性
            const updateInterval = 100; // UI更新间隔
            let lastUpdateTime = Date.now();

            // 预处理所有词语，过滤掉重复的
            const existingTexts = new Set(targetHighlights.map(h => h.text.toLowerCase()));
            const newWords = words.filter(word => !existingTexts.has(word.toLowerCase()));

            if (newWords.length === 0) {
                alert('所有词语都已存在，未添加新的高亮词。');
                return;
            }

            // 显示进度条
            const progressDiv = document.createElement('div');
            progressDiv.style.cssText = `
                position: fixed;
                top: 0;
                left: 0;
                right: 0;
                height: 4px;
                background: #f0f0f0;
                z-index: 10000;
            `;
            const progressBar = document.createElement('div');
            progressBar.style.cssText = `
                width: 0%;
                height: 100%;
                background: #1a73e8;
                transition: width 0.3s ease;
            `;
            progressDiv.appendChild(progressBar);
            document.body.appendChild(progressDiv);

            // 分批处理新词语
            for (let i = 0; i < newWords.length; i += batchSize) {
                const batch = newWords.slice(i, i + batchSize);
                const batchHighlights = batch.map(word => ({
                    text: word,
                    color: highlightColor.value,
                    id: Date.now() + '-' + Math.random().toString(36).substr(2, 9),
                    caseSensitive: settings.caseSensitive,
                    globalMatch: true,
                    processAll: true,
                    multipleMatches: true,
                    maintainAllHighlights: true
                }));

                targetHighlights.push(...batchHighlights);
                addedCount += batchHighlights.length;

                // 更新进度条
                const progress = ((i + batchSize) / newWords.length) * 100;
                progressBar.style.width = `${Math.min(progress, 100)}%`;

                // 优化UI更新频率
                const currentTime = Date.now();
                if (currentTime - lastUpdateTime >= updateInterval) {
                    // 保存到local存储
                    await new Promise((resolve, reject) => {
                        chrome.storage.local.set({ highlights }, (result) => {
                            if (chrome.runtime.lastError) {
                                reject(chrome.runtime.lastError);
                            } else {
                                resolve(result);
                            }
                        });
                    });

                    // 更新UI显示
                    updateHighlightList(targetHighlights);
                    lastUpdateTime = currentTime;
                }

                // 添加小延迟，让UI有机会更新
                await new Promise(resolve => setTimeout(resolve, 10));
            }

            // 确保最后一次更新
            await new Promise((resolve, reject) => {
                chrome.storage.local.set({ highlights }, (result) => {
                    if (chrome.runtime.lastError) {
                        reject(chrome.runtime.lastError);
                    } else {
                        resolve(result);
                    }
                });
            });

            // 更新UI显示
            updateHighlightList(targetHighlights);

            // 移除进度条
            document.body.removeChild(progressDiv);

            // 显示导入结果
            alert(`成功导入 ${addedCount} 个高亮词！`);

            // 如果启用了自动高亮，应用高亮
            if (settings.autoHighlight && addedCount > 0) {
                chrome.tabs.query({active: true, currentWindow: true}, function(tabs) {
                    if (tabs[0]) {
                        chrome.tabs.sendMessage(tabs[0].id, {
                            action: 'updateHighlights',
                            enabled: true,
                            force: true,
                            highlights: targetHighlights.map(h => ({
                                ...h,
                                globalMatch: true,
                                processAll: true,
                                multipleMatches: true,
                                maintainAllHighlights: true
                            })),
                            settings: {
                                ...settings,
                                globalMatch: true,
                                batchSize: 50,
                                processingInterval: 100,
                                processAll: true,
                                multipleMatches: true,
                                maintainAllHighlights: true,
                                reprocessExisting: true,
                                useHardwareAcceleration: true,
                                usePositionLocking: true,
                                renderMode: 'stable'
                            }
                        });
                    }
                });
            }
        } catch (error) {
            console.error('导入失败:', error);
            alert(`导入失败: ${error.message || '未知错误'}`);
            
            // 确保移除进度条
            const progressDiv = document.querySelector('div[style*="z-index: 10000"]');
            if (progressDiv && progressDiv.parentNode) {
                progressDiv.parentNode.removeChild(progressDiv);
            }
        }
    }

    // Export functionality
    exportBtn.addEventListener('click', async function() {
        try {
            const data = await new Promise(resolve => chrome.storage.local.get(['highlights', 'settings'], resolve));
            const settings = data.settings ?? {};
            
            // 根据onlyCurrentTab设置选择正确的高亮词来源
            let highlights = [];
            if (settings.onlyCurrentTab) {
                // 获取当前标签页ID
                const tabs = await new Promise(resolve => {
                    chrome.tabs.query({active: true, currentWindow: true}, resolve);
                });
                if (tabs && tabs.length > 0) {
                    const currentTabId = tabs[0].id.toString();
                    highlights = data.highlights?.tabs?.[currentTabId] ?? [];
                } else {
                    alert('无法获取当前标签页信息');
                    return;
                }
            } else {
                highlights = data.highlights?.default ?? [];
            }
            
            if (highlights.length === 0) {
                alert('没有可导出的高亮词！');
                return;
            }

            const content = highlights.map(h => h.text).join('\n');
            const blob = new Blob([content], { type: 'text/plain;charset=utf-8' });
            const url = URL.createObjectURL(blob);
            
            const a = document.createElement('a');
            a.href = url;
            a.download = '高亮词列表.txt';
            document.body.appendChild(a);
            a.click();
            
            setTimeout(() => {
                document.body.removeChild(a);
                URL.revokeObjectURL(url);
            }, 0);
        } catch (error) {
            console.error('导出失败:', error);
            alert('导出失败，请重试！');
        }
    });

    // Update tab switching functionality
    document.querySelectorAll('.nav-tab').forEach(tab => {
        tab.addEventListener('click', function(e) {
            e.preventDefault();
            const tabId = this.dataset.tab;
            
            // Remove all active states
            document.querySelectorAll('.nav-tab').forEach(t => t.classList.remove('active'));
            document.querySelectorAll('.tab-pane').forEach(p => {
                p.classList.remove('active');
            });
            
            // Set new active tab
            this.classList.add('active');
            const pane = document.getElementById(tabId);
            if (pane) {
                pane.classList.add('active');
            }
        });
    });

    // Initialize default tab
    document.addEventListener('DOMContentLoaded', async () => {
        try {
            // 从local存储加载数据
            const data = await new Promise(resolve => chrome.storage.local.get(['highlights', 'settings', 'enabled'], resolve));
            const highlights = data.highlights ?? { default: [] };
            const settings = data.settings ?? {};
            const enabled = data.enabled ?? true;

            // 更新UI显示
            updateHighlightList(highlights.default);

            // 初始化默认标签页
            const homeTab = document.getElementById('home');
            if (homeTab) {
                homeTab.classList.add('active');
            }

            // 初始化设置
            const enableHighlight = document.getElementById('enableHighlight');
            const autoHighlight = document.getElementById('autoHighlight');
            const caseSensitive = document.getElementById('caseSensitive');
            const globalMatch = document.getElementById('globalMatch');

            enableHighlight.checked = enabled;
            autoHighlight.checked = settings.autoHighlight ?? true;
            caseSensitive.checked = settings.caseSensitive ?? false;
            globalMatch.checked = settings.globalMatch ?? true;

            // 确保设置被保存
            const updatedSettings = {
                ...settings,
                autoHighlight: autoHighlight.checked,
                caseSensitive: caseSensitive.checked,
                globalMatch: globalMatch.checked,
                enabled: enabled
            };

            await chrome.storage.local.set({ settings: updatedSettings });

            // 如果启用了高亮和自动高亮，立即应用高亮
            if (enabled && updatedSettings.autoHighlight && highlights.default.length > 0) {
                chrome.tabs.query({active: true, currentWindow: true}, function(tabs) {
                    if (tabs[0]) {
                        chrome.tabs.sendMessage(tabs[0].id, {
                            action: 'updateHighlights',
                            enabled: true,
                            force: true,
                            highlights: highlights.default.map(h => ({
                                ...h,
                                globalMatch: true,
                                processAll: true,
                                multipleMatches: true,
                                maintainAllHighlights: true
                            })),
                            settings: {
                                ...updatedSettings,
                                globalMatch: true,
                                batchSize: 50,
                                processingInterval: 100,
                                processAll: true,
                                multipleMatches: true,
                                maintainAllHighlights: true,
                                reprocessExisting: true
                            }
                        });
                    }
                });
            }

        } catch (error) {
            console.error('初始化失败:', error);
        }
    });

    // Add settings change handlers
    document.getElementById('highlightDelay').addEventListener('change', async function() {
        const delay = parseInt(this.value);
        if (isNaN(delay) || delay < 0) return;

        const data = await chrome.storage.local.get(['settings']);
        const settings = data.settings || {};
        settings.highlightDelay = delay;
        await updateSettings(settings);
        throttledUpdate();
    });

    document.getElementById('renderMode').addEventListener('change', function() {
        const mode = this.value;
        
        chrome.storage.local.get(['settings'], function(data) {
            const settings = data.settings || {};
            settings.renderMode = mode;
            chrome.storage.local.set({ settings }, function() {
                throttledUpdate();
            });
        });
    });

    document.getElementById('useHardwareAcceleration').addEventListener('change', function() {
        const enabled = this.checked;
        
        chrome.storage.local.get(['settings'], function(data) {
            const settings = data.settings || {};
            settings.useHardwareAcceleration = enabled;
            chrome.storage.local.set({ settings }, function() {
                throttledUpdate();
            });
        });
    });

    document.getElementById('usePositionLocking').addEventListener('change', function() {
        const enabled = this.checked;
        
        chrome.storage.local.get(['settings'], function(data) {
            const settings = data.settings || {};
            settings.usePositionLocking = enabled;
            chrome.storage.local.set({ settings }, function() {
                throttledUpdate();
            });
        });
    });

    // Add privacy policy and permission list modals
    const privacyPolicyContent = `
        <div class="modal-content privacy-content">
            <button class="modal-close">&times;</button>
            <h3>隐私政策</h3>
            
            <section>
                <h4>1. 信息收集</h4>
                <p>我们收集以下类型的信息：</p>
                <ul>
                    <li>用户创建的高亮词列表</li>
                    <li>用户的设置偏好</li>
                    <li>基本的使用统计信息</li>
                </ul>
            </section>

            <section>
                <h4>2. 信息使用</h4>
                <p>收集的信息将用于：</p>
                <ul>
                    <li>提供和改进高亮功能</li>
                    <li>保存用户的个性化设置</li>
                    <li>优化插件性能</li>
                    <li>提供技术支持</li>
                </ul>
            </section>

            <section>
                <h4>3. 数据存储</h4>
                <p>所有数据均存储在：</p>
                <ul>
                    <li>用户浏览器的本地存储中</li>
                    <li>Chrome同步存储中（如果用户启用了同步）</li>
                </ul>
                <p>我们不会将数据上传至任何外部服务器。</p>
            </section>

            <section>
                <h4>4. 数据安全</h4>
                <ul>
                    <li>所有数据仅存储在用户的设备上</li>
                    <li>使用Chrome提供的安全存储API</li>
                    <li>不与第三方共享任何数据</li>
                </ul>
            </section>

            <section>
                <h4>5. 用户权利</h4>
                <ul>
                    <li>随时查看和修改已保存的高亮词</li>
                    <li>完全控制插件的启用/禁用</li>
                    <li>可以随时清除所有保存的数据</li>
                    <li>导出和导入数据的权利</li>
                </ul>
            </section>

            <section>
                <h4>6. 更新通知</h4>
                <p>当本隐私政策发生实质性变更时，我们将：</p>
                <ul>
                    <li>通过插件界面通知用户</li>
                    <li>要求用户重新确认同意</li>
                    <li>提供更改内容的摘要</li>
                </ul>
            </section>
        </div>
    `;

    const permissionListContent = `
        <div class="modal-content permission-content">
            <button class="modal-close">&times;</button>
            <h3>权限清单</h3>

            <section>
                <h4>1. 必需权限</h4>
                <div class="permission-item">
                    <h5>存储权限</h5>
                    <ul>
                        <li>存储用户的高亮词列表</li>
                        <li>保存用户的设置选项</li>
                        <li>存储主题和界面偏好</li>
                    </ul>
                </div>

                <div class="permission-item">
                    <h5>标签页访问权限</h5>
                    <ul>
                        <li>读取当前页面内容以实现高亮功能</li>
                        <li>监听页面变化以更新高亮</li>
                        <li>在页面中注入必要的样式</li>
                    </ul>
                </div>
            </section>

            <section>
                <h4>2. 收集的信息</h4>
                <div class="data-item">
                    <h5>用户数据</h5>
                    <ul>
                        <li>高亮词列表及其属性</li>
                        <li>颜色设置</li>
                        <li>匹配规则设置</li>
                    </ul>
                </div>

                <div class="data-item">
                    <h5>设置数据</h5>
                    <ul>
                        <li>渲染模式选择</li>
                        <li>高亮延迟设置</li>
                        <li>性能优化选项</li>
                    </ul>
                </div>

                <div class="data-item">
                    <h5>使用数据</h5>
                    <ul>
                        <li>功能使用频率统计</li>
                        <li>性能相关数据</li>
                        <li>错误报告信息</li>
                    </ul>
                </div>
            </section>

            <section>
                <h4>3. 数据使用说明</h4>
                <ul>
                    <li>所有数据仅用于提供和改进插件功能</li>
                    <li>不会上传至外部服务器</li>
                    <li>不会用于商业目的</li>
                    <li>不会与第三方共享</li>
                </ul>
            </section>
        </div>
    `;

    const updateLogContent = `
        <div class="modal-content update-log-content">
            <button class="modal-close">&times;</button>
            <h3>更新日志</h3>
            
            <section>
                <h4><strong>v2.0.6</strong> (2025-06-25)</h4>
                <ul>
                    <li>修复了部分已知问题</li>
                    <li>增加了只在当前标签页应用功能（未完善可能有问题）</li>
                </ul>
            </section>
            
            <section>
                <h4><strong>v2.0.5</strong> (2025-04-25)</h4>
                <ul>
                    <li>修复了部分已知问题</li>
                    <li>增加了建议与反馈入口</li>
                </ul>
            </section>
            
            <section>
                <h4><strong>v2.0.4</strong> (2025-03-30)</h4>
                <ul>
                    <li>修复了部分已知问题</li>
                </ul>
            </section>
            
            <section>
                <h4><strong>v2.0.3</strong> (2025-03-20)</h4>
                <ul>
                    <li>修复了复选框对钩不居中问题</li>
                    <li>添加导入高亮词上限总数由2000个修改为100000个</li>
                    <li>修复了导入大量高亮词后插件变卡的问题</li>
                </ul>
            </section>
            
            <section>
                <h4><strong>v2.0.2</strong> (2025-03-11)</h4>
                <ul>
                    <li>修复了无法导入超过50个高亮词的问题</li>
                    <li>增加了高亮词计数显示</li>
                    <li>修复了删除、增加、修改高亮词实时显示状态问题</li>
                    <li>修复了插件系统错误</li>
                    <li>设置功能（待开发）</li>
                </ul>
            </section>
            
            <section>
                <h4><strong>v2.0.1</strong> (2025-03-10)</h4>
                <ul>
                    <li>增加了导出高亮词功能</li>
                    <li>重构插件界面UI</li>
                    <li>新增底部导航栏和相应信息显示</li>
                    <li>增加了隐私政策和权限信息</li>
                </ul>
            </section>
            
            <section>
                <h4><strong>v1.0.0</strong> (2025-03-08)</h4>
                <ul>
                    <li>插件第一个版本支持添加、导入、删除、修改高亮词</li>
                    <li>支持高亮词颜色修改</li>
                    <li>支持清除全部高亮词</li>
                    <li>支持自动高亮词</li>
                    <li>支持区分大小写（需要刷新页面生效：待修复）</li>
                    <li>支持全局匹配功能</li>
                    <li>支持多选和全选高亮词进行删除操作</li>
                </ul>
            </section>
        </div>
    `;

    // Add privacy policy and permission list button event listeners
    document.getElementById('privacyPolicy').addEventListener('click', function() {
        const modal = document.createElement('div');
        modal.className = 'modal';
        modal.innerHTML = privacyPolicyContent;
        document.body.appendChild(modal);
        modal.style.display = 'block';

        modal.querySelector('.modal-close').addEventListener('click', function() {
            document.body.removeChild(modal);
        });
    });

    document.getElementById('permissionList').addEventListener('click', function() {
        const modal = document.createElement('div');
        modal.className = 'modal';
        modal.innerHTML = permissionListContent;
        document.body.appendChild(modal);
        modal.style.display = 'block';

        modal.querySelector('.modal-close').addEventListener('click', function() {
            document.body.removeChild(modal);
        });
    });
    
    document.getElementById('updateLog').addEventListener('click', function() {
        const modal = document.createElement('div');
        modal.className = 'modal';
        modal.innerHTML = updateLogContent;
        document.body.appendChild(modal);
        modal.style.display = 'block';

        modal.querySelector('.modal-close').addEventListener('click', function() {
            document.body.removeChild(modal);
        });
    });

    // Add highlight word with enter key support
    highlightWord.addEventListener('keypress', function(e) {
        if (e.key === 'Enter') {
            e.preventDefault();
            addWordBtn.click();
        }
    });

    // Add search functionality
    const searchHighlight = document.getElementById('searchHighlight');
    searchHighlight.addEventListener('input', function() {
        const searchTerm = this.value.toLowerCase().trim();
        
        chrome.storage.local.get(['highlights', 'settings'], function(data) {
            const highlights = data.highlights ?? {};
            highlights.default = highlights.default ?? [];
            highlights.tabs = highlights.tabs ?? {};
            const settings = data.settings ?? {};
            
            let targetHighlights = [];
            
            if (settings.onlyCurrentTab) {
                // 获取当前标签页的高亮词
                chrome.tabs.query({active: true, currentWindow: true}, function(tabs) {
                    if (tabs && tabs.length > 0) {
                        const currentTabId = tabs[0].id.toString();
                        let tabHighlights = highlights.tabs[currentTabId] ?? [];
                        
                        // 如果tabs中没有高亮词，则尝试从currentHighlights获取
                        if (tabHighlights.length === 0) {
                            chrome.storage.local.get(['currentHighlights'], function(result) {
                                const currentHighlights = result.currentHighlights ?? [];
                                console.debug('搜索时从currentHighlights获取高亮词:', currentHighlights.length);
                                targetHighlights = currentHighlights;
                                
                                // 同步到tabs存储，确保数据一致性
                                if (currentHighlights.length > 0) {
                                    if (!highlights.tabs) highlights.tabs = {};
                                    highlights.tabs[currentTabId] = currentHighlights;
                                    chrome.storage.local.set({ highlights: highlights }, function() {
                                        console.debug('已同步currentHighlights到tabs存储');
                                    });
                                }
                                
                                processSearch(targetHighlights, searchTerm);
                            });
                        } else {
                            targetHighlights = tabHighlights;
                            
                            // 同步到currentHighlights，确保数据一致性
                            chrome.storage.local.set({ currentHighlights: tabHighlights }, function() {
                                console.debug('已同步tabs存储到currentHighlights');
                            });
                            
                            processSearch(targetHighlights, searchTerm);
                        }
                    }
                });
            } else {
                // 使用全局高亮词
                targetHighlights = highlights.default;
                processSearch(targetHighlights, searchTerm);
            }
            
            function processSearch(highlightList, term) {
                // If search term is empty, show all highlights
                if (!term) {
                    updateHighlightList(highlightList);
                    // 更新勾选数量显示
                    const checkedBoxes = highlightContainer.querySelectorAll('.checkbox:checked');
                    updateHighlightCountDisplay(highlightList.length, highlightList.length, checkedBoxes.length);
                    return;
                }
                
                // Filter highlights based on search term
                const filteredHighlights = highlightList.filter(highlight => 
                    highlight.text.toLowerCase().includes(term)
                );
                
                updateHighlightList(filteredHighlights);
                
                // 更新勾选数量显示
                setTimeout(() => {
                    const checkedBoxes = highlightContainer.querySelectorAll('.checkbox:checked');
                    updateHighlightCountDisplay(highlightList.length, filteredHighlights.length, checkedBoxes.length);
                }, 0);
            }
        });
    });

    // Update search icon styles
    const searchStyles = document.createElement('style');
    searchStyles.textContent = `
        .search-group {
            position: relative;
            margin-bottom: 0;
            background: white;
            border-radius: 8px;
            padding: 14px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.05);
        }

        .search-icon {
            position: absolute;
            left: 24px;
            top: 50%;
            transform: translateY(-50%);
            color: #94a3b8;
            pointer-events: none;
            z-index: 1;
        }

        #searchHighlight {
            width: 100%;
            padding: 8px 14px 8px 36px;
            border: 2px solid #e2e8f0;
            border-radius: 8px;
            font-size: 14px;
            background: white;
            transition: all 0.2s ease;
            position: relative;
        }

        #searchHighlight:focus {
            border-color: #1a73e8;
            box-shadow: 0 0 0 3px rgba(26, 115, 232, 0.1);
        }

        #searchHighlight:focus + .search-icon {
            color: #1a73e8;
        }

        .highlight-item .text mark {
            background-color: #fef08a;
            color: #1a73e8;
            padding: 0 2px;
            border-radius: 2px;
            font-weight: 500;
        }

        .no-results {
            padding: 20px;
            text-align: center;
            color: #64748b;
            font-style: italic;
        }
    `;
    document.head.appendChild(searchStyles);
    
    // 添加模态窗口的通用样式
    const modalStyles = document.createElement('style');
    modalStyles.textContent = `
        .modal {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.5);
            z-index: 1000;
            display: flex;
            justify-content: center;
            align-items: center;
        }
        
        .modal-content {
            background-color: white;
            border-radius: 8px;
            padding: 20px;
            width: 90%;
            max-width: 500px;
            max-height: 80vh;
            overflow-y: auto;
            position: relative;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        
        .modal-close {
            position: absolute;
            top: 10px;
            right: 10px;
            background: none;
            border: none;
            font-size: 24px;
            cursor: pointer;
            color: #64748b;
        }
        
        .modal-close:hover {
            color: #1a73e8;
        }
        
        .modal-content h3 {
            margin-top: 0;
            padding-bottom: 10px;
            border-bottom: 1px solid #e2e8f0;
            color: #1a73e8;
        }
        
        .modal-content section {
            margin-bottom: 20px;
        }
        
        .modal-content h4 {
            margin-bottom: 10px;
            color: #334155;
        }
        
        .modal-content ul {
            margin: 0;
            padding-left: 20px;
        }
        
        .modal-content li {
            margin-bottom: 5px;
        }
        
        .update-log-content section {
            border-bottom: 1px solid #f1f5f9;
            padding-bottom: 15px;
        }
        
        .update-log-content section:last-child {
            border-bottom: none;
        }
        
        .update-log-content h4 {
            color: #1e40af;
            font-weight: 600;
        }
    `;
    document.head.appendChild(modalStyles);

    // 修改自动高亮按钮的处理
    autoHighlight.addEventListener('change', async function() {
        const isChecked = this.checked;
        const data = await chrome.storage.local.get(['highlights', 'settings', 'enabled']);
        const settings = data.settings ?? {};
        const highlights = data.highlights?.default ?? [];
        const enabled = data.enabled ?? true;

        // 更新设置
        const updatedSettings = {
            ...settings,
            autoHighlight: isChecked
        };
        
        await chrome.storage.local.set({ settings: updatedSettings });

        // 只有当插件启用时才应用高亮
        if (enabled) {
            chrome.tabs.query({active: true, currentWindow: true}, function(tabs) {
                if (tabs[0]) {
                    chrome.tabs.sendMessage(tabs[0].id, {
                        action: isChecked ? 'updateHighlights' : 'removeHighlights',
                        enabled: true,
                        force: true,
                        highlights: highlights.map(h => ({
                            ...h,
                            globalMatch: true,
                            processAll: true,
                            multipleMatches: true,
                            maintainAllHighlights: true
                        })),
                        settings: {
                            ...updatedSettings,
                            globalMatch: true,
                            batchSize: 50,
                            processingInterval: 100,
                            processAll: true,
                            multipleMatches: true,
                            maintainAllHighlights: true,
                            reprocessExisting: true
                        }
                    });
                }
            });
        }
    });

    // 修改设置更新逻辑
    function updateSettings(settings) {
        return new Promise((resolve, reject) => {
            chrome.storage.local.set({ settings }, resolve);
        });
    }

    // 更新高亮词数量显示函数
    function updateHighlightCountDisplay(totalCount, displayCount, selectedCount) {
        // 如果高亮词计数元素不存在，则创建
        let countElement = document.getElementById('highlightCountDisplay');
        if (!countElement) {
            countElement = document.createElement('div');
            countElement.id = 'highlightCountDisplay';
            countElement.className = 'highlight-count-display';
            
            // 插入到highlight-header后面
            const headerElement = document.querySelector('.highlight-header');
            if (headerElement && headerElement.nextSibling) {
                headerElement.parentNode.insertBefore(countElement, headerElement.nextSibling);
            } else if (headerElement) {
                headerElement.parentNode.appendChild(countElement);
            }
            
            // 添加相关CSS
            const style = document.createElement('style');
            style.textContent = `
                .highlight-count-display {
                    display: flex;
                    justify-content: space-between;
                    padding: 5px 10px;
                    font-size: 13px;
                    color: #4b5563;
                    background-color: #f3f4f6;
                    border-radius: 4px;
                    margin: 5px 0;
                }
                .highlight-count-item {
                    display: flex;
                    align-items: center;
                }
                .highlight-count-item .count-icon {
                    margin-right: 5px;
                    font-size: 14px;
                }
            `;
            document.head.appendChild(style);
        }
        
        // 更新内容
        countElement.innerHTML = `
            <div class="highlight-count-item">
                <span class="count-icon">📊</span>
                <span>总计: ${totalCount}个高亮词</span>
            </div>
            ${displayCount !== totalCount ? 
                `<div class="highlight-count-item">
                    <span class="count-icon">🔍</span>
                    <span>显示: ${displayCount}个</span>
                </div>` : ''
            }
            ${selectedCount > 0 ? 
                `<div class="highlight-count-item">
                    <span class="count-icon">✓</span>
                    <span>已选: ${selectedCount}个</span>
                </div>` : ''
            }
        `;
    }

    // 添加扩展上下文检查函数
    function isExtensionContextValid() {
        try {
            // 尝试访问chrome.runtime.id - 如果扩展上下文无效，这会抛出错误
            return !!chrome.runtime.id;
        } catch (e) {
            console.debug('扩展上下文已失效', e);
            return false;
        }
    }

    // 添加安全的消息发送函数
    async function sendMessageSafely(tabId, message) {
        if (!isExtensionContextValid()) {
            throw new Error('Extension context invalidated');
        }
        
        return new Promise((resolve, reject) => {
            try {
                chrome.tabs.sendMessage(tabId, message, (response) => {
                    if (chrome.runtime.lastError) {
                        // 捕获并记录错误，但不必拒绝Promise
                        console.debug('发送消息时出错:', chrome.runtime.lastError.message);
                        // 返回错误对象而不是抛出错误
                        resolve({ 
                            success: false, 
                            error: chrome.runtime.lastError.message,
                            errorType: 'sendMessageError' 
                        });
                    } else {
                        resolve(response || { success: true });
                    }
                });
            } catch (err) {
                console.error('发送消息时发生异常:', err);
                resolve({ 
                    success: false, 
                    error: err.message || JSON.stringify(err),
                    errorType: 'exception' 
                });
            }
        });
    }

    // 检查内容脚本是否准备好
    async function checkContentScriptReady(tabId) {
        if (!isExtensionContextValid()) return false;
        
        try {
            const response = await sendMessageSafely(tabId, { action: 'ping' });
            return response.success;
        } catch (err) {
            console.debug('检查内容脚本状态时出错:', err);
            return false;
        }
    }

    // 改进的更新高亮函数
    async function updateHighlights(force = false) {
        if (!isExtensionContextValid()) {
            console.debug('更新高亮时扩展上下文已失效');
            return;
        }
        
        try {
            const tabsResult = await new Promise((resolve) => {
                chrome.tabs.query({active: true, currentWindow: true}, resolve);
            });
            
            if (!tabsResult || !tabsResult.length || !tabsResult[0] || !tabsResult[0].id) {
                console.debug('没有可用的活动标签页');
                return;
            }
            
            const activeTab = tabsResult[0];
            
            // 只发送消息到http/https页面
            if (!activeTab.url || !activeTab.url.match(/^https?:\/\//)) {
                console.debug('不支持当前页面类型:', activeTab.url);
                return;
            }
            
            // 获取存储的数据
            const data = await new Promise((resolve) => {
                chrome.storage.local.get(['highlights', 'settings'], resolve);
            });
            
            // 根据onlyCurrentTab设置选择正确的高亮词来源
            let highlights = [];
            const storedSettings = data.settings || {};
            if (storedSettings.onlyCurrentTab) {
                const currentTabId = activeTab.id.toString();
                highlights = data.highlights?.tabs?.[currentTabId] ?? [];
            } else {
                highlights = data.highlights?.default ?? [];
            }
            const settings = {
                renderMode: document.getElementById('renderMode')?.value || 'stable',
                highlightDelay: parseInt(document.getElementById('highlightDelay')?.value || '800'),
                useHardwareAcceleration: document.getElementById('useHardwareAcceleration')?.checked || true,
                usePositionLocking: document.getElementById('usePositionLocking')?.checked || true,
                batchSize: 50,
                processingInterval: 100,
                processAll: true,
                globalMatch: true,
                reprocessExisting: true,
                multipleMatches: true,
                maintainAllHighlights: true,
                autoHighlight: true,
                // 添加防闪烁相关设置
                preventFlickering: true,
                useStableRendering: true,
                renderInBatches: true,
                batchRenderSize: 100,
                renderTimeout: 50,
                useRAF: true,
                useIntersectionObserver: true,
                maintainHighlightPositions: true,
                useCSSContainment: true,
                useWillChange: true,
                useTransform3D: true,
                useBackfaceVisibility: true,
                usePerspective: true
            };
            
            // 先检查内容脚本是否准备好
            const isReady = await checkContentScriptReady(activeTab.id);
            if (!isReady) {
                console.debug('内容脚本未准备好，可能是新页面或不兼容的页面');
                return;
            }
            
            // 使用防闪烁策略移除高亮
            const removeResult = await sendMessageSafely(activeTab.id, {
                action: 'removeHighlights',
                force: true,
                maintain: true,
                processAll: true,
                multipleMatches: true,
                maintainAllHighlights: true,
                preventFlickering: true,
                useStableRendering: true,
                highlightTexts: highlights.map(h => h.text)
            });
            
            if (!removeResult.success) {
                console.debug('移除高亮失败:', removeResult.error);
            }
            
            // 确保扩展上下文仍然有效
            if (!isExtensionContextValid()) return;
            
            // 使用requestAnimationFrame优化渲染时机
            await new Promise(resolve => {
                requestAnimationFrame(() => {
                    setTimeout(resolve, settings.renderTimeout);
                });
            });
            
            // 分批应用高亮
            const batchSize = settings.batchRenderSize;
            for (let i = 0; i < highlights.length; i += batchSize) {
                const batch = highlights.slice(i, i + batchSize);
                
                // 重新应用当前批次的高亮
                const updateResult = await sendMessageSafely(activeTab.id, {
                    action: 'updateHighlights',
                    force: true,
                    maintain: true,
                    highlights: batch.map(h => ({
                        ...h,
                        globalMatch: true,
                        processAll: true,
                        multipleMatches: true,
                        maintainAllHighlights: true,
                        preventFlickering: true,
                        useStableRendering: true,
                        useRAF: true,
                        useIntersectionObserver: true,
                        maintainHighlightPositions: true,
                        useCSSContainment: true,
                        useWillChange: true,
                        useTransform3D: true,
                        useBackfaceVisibility: true,
                        usePerspective: true
                    })),
                    settings: settings
                });
                
                if (!updateResult.success) {
                    console.debug('应用高亮批次失败:', updateResult.error);
                }
                
                // 等待下一帧
                await new Promise(resolve => {
                    requestAnimationFrame(() => {
                        setTimeout(resolve, settings.renderTimeout);
                    });
                });
            }
        } catch (error) {
            console.error('更新高亮过程中出错:', error);
        }
    }

    // 添加反馈按钮点击事件处理
    document.getElementById('feedbackBtn').addEventListener('click', function() {
        window.open('https://jdadtq277s.feishu.cn/share/base/form/shrcnb1q9cQ8qUXHYiKSiQhBL3e', '_blank');
    });
}); // DOMContentLoaded 事件处理函数结束