

// 存储原始的requestFullscreen方法
const originalMethods = {};

// 全屏API方法名数组（兼容不同浏览器）
const fullscreenMethods = [
    'requestFullscreen',
    'webkitRequestFullscreen',
    'mozRequestFullScreen',
    'msRequestFullscreen'
];

// 延迟时间（毫秒）
const DELAY_TIME = 20;

// 防止循环调用的标志
let isExecutingFullscreen = false;

// 重写HTMLElement原型上的全屏方法
function overrideFullscreenMethods() {
    fullscreenMethods.forEach(method => {
        if (HTMLElement.prototype[method]) {
            // 保存原始方法
            originalMethods[method] = HTMLElement.prototype[method];

            // 重写方法
            HTMLElement.prototype[method] = function (options) {
                const element = this;
                const originalMethod = originalMethods[method];

                // 防止循环调用
                if (isExecutingFullscreen) {
                    //console.log(`[Video Fullscreen Delay] 检测到循环调用，直接执行原始方法`);
                    return originalMethod.call(element, options);
                }

                //console.log(`[Video Fullscreen Delay] 拦截到全屏请求，将在${DELAY_TIME}ms后执行`);

                // 延迟执行原始的全屏方法
                setTimeout(() => {
                    try {
                        isExecutingFullscreen = true;
                        originalMethod.call(element, options);
                        //console.log(`[Video Fullscreen Delay] 延迟全屏已执行`);
                    } catch (error) {
                        console.error('[Video Fullscreen Delay] 全屏执行出错:', error);
                    } finally {
                        // 延迟重置标志，确保其他调用能正常工作
                        setTimeout(() => {
                            isExecutingFullscreen = false;
                        }, 100);
                    }
                }, DELAY_TIME);
            };
        }
    });
}

// 监听双击事件（针对双击全屏的情况）
function handleDoubleClick(event) {
    const target = event.target;

    // 检查当前是否已经处于全屏状态
    const isCurrentlyFullscreen = !!(
        document.fullscreenElement ||
        document.webkitFullscreenElement ||
        document.mozFullScreenElement ||
        document.msFullscreenElement
    );

    // 检查是否是视频元素或视频容器
    if (target.tagName === 'VIDEO' ||
        target.closest('video') ||
        target.classList.contains('video-player') ||
        target.querySelector('video')) {

        if (isCurrentlyFullscreen) {
            // 如果已经是全屏状态，双击应该退出全屏（延迟执行）
            //console.log('[Video Fullscreen Delay] 检测到全屏状态下的双击，将延迟退出全屏');

            event.preventDefault();
            event.stopPropagation();

            setTimeout(() => {
                try {
                    // 再次检查是否还在全屏状态
                    const stillFullscreen = !!(
                        document.fullscreenElement ||
                        document.webkitFullscreenElement ||
                        document.mozFullScreenElement ||
                        document.msFullscreenElement
                    );

                    if (stillFullscreen) {
                        // 退出全屏
                        if (document.exitFullscreen) {
                            document.exitFullscreen().catch(err => {
                                console.warn('[Video Fullscreen Delay] 退出全屏失败:', err.message);
                            });
                        } else if (document.webkitExitFullscreen) {
                            document.webkitExitFullscreen();
                        } else if (document.mozCancelFullScreen) {
                            document.mozCancelFullScreen();
                        } else if (document.msExitFullscreen) {
                            document.msExitFullscreen();
                        }
                    } else {
                        //console.log('[Video Fullscreen Delay] 已不在全屏状态，跳过退出操作');
                    }
                } catch (error) {
                    console.warn('[Video Fullscreen Delay] 退出全屏时出错:', error.message);
                }
            }, DELAY_TIME);
        } else {
            // 如果不是全屏状态，双击进入全屏（延迟执行）
            //console.log('[Video Fullscreen Delay] 检测到视频区域双击，将延迟进入全屏');

            event.preventDefault();
            event.stopPropagation();

            setTimeout(() => {
                // 尝试找到视频元素
                let videoElement = target.tagName === 'VIDEO' ? target : target.closest('video') || target.querySelector('video');

                if (videoElement) {
                    // 直接调用原始方法，避免被拦截
                    const method = fullscreenMethods.find(m => originalMethods[m] && videoElement[m]);
                    if (method) {
                        isExecutingFullscreen = true;
                        originalMethods[method].call(videoElement);
                        setTimeout(() => {
                            isExecutingFullscreen = false;
                        }, 100);
                    }
                } else {
                    // 如果没找到视频元素，尝试对整个容器全屏
                    const method = fullscreenMethods.find(m => originalMethods[m] && target[m]);
                    if (method) {
                        isExecutingFullscreen = true;
                        originalMethods[method].call(target);
                        setTimeout(() => {
                            isExecutingFullscreen = false;
                        }, 100);
                    }
                }
            }, DELAY_TIME);
            restartAudioCapture();
        }
    }
}

// 监听全屏按钮点击事件
function interceptFullscreenButtons() {
    // 常见的全屏按钮选择器
    const fullscreenSelectors = [
        '[title*="全屏"]',
        '[title*="fullscreen"]',
        '[title*="Fullscreen"]',
        '[aria-label*="全屏"]',
        '[aria-label*="fullscreen"]',
        '[aria-label*="Fullscreen"]',
        '.fullscreen-button',
        '.video-fullscreen',
        '.vp-fullscreen',
        '.fullscreen-icon',
        '[data-title*="全屏"]',
        '[data-tooltip*="全屏"]'
    ];

    // 退出全屏按钮选择器
    const exitFullscreenSelectors = [
        '[title*="退出全屏"]',
        '[title*="exit fullscreen"]',
        '[title*="Exit Fullscreen"]',
        '[aria-label*="退出全屏"]',
        '[aria-label*="exit fullscreen"]',
        '[aria-label*="Exit Fullscreen"]',
        '.exit-fullscreen-button',
        '.video-exit-fullscreen'
    ];

    // 用于控制事件监听器的标志
    let isListenerActive = true;

    function clickHandler(event) {
        if (!isListenerActive) return;

        const target = event.target;

        // 检查当前是否已经处于全屏状态
        const isCurrentlyFullscreen = !!(
            document.fullscreenElement ||
            document.webkitFullscreenElement ||
            document.mozFullScreenElement ||
            document.msFullscreenElement
        );

        const isFullscreenButton = fullscreenSelectors.some(selector => {
            try {
                return target.matches(selector) || target.closest(selector);
            } catch (e) {
                return false;
            }
        });

        const isExitFullscreenButton = exitFullscreenSelectors.some(selector => {
            try {
                return target.matches(selector) || target.closest(selector);
            } catch (e) {
                return false;
            }
        });

        // 检查是否点击了全屏相关的按钮
        const isFullscreenRelated = isFullscreenButton || isExitFullscreenButton ||
            target.innerHTML.includes('全屏') ||
            target.innerHTML.includes('退出全屏') ||
            target.className.toLowerCase().includes('fullscreen') ||
            target.getAttribute('title')?.includes('全屏') ||
            target.getAttribute('aria-label')?.includes('全屏');

        if (isFullscreenRelated) {
            if (isCurrentlyFullscreen && (isExitFullscreenButton || target.innerHTML.includes('退出全屏'))) {
                //console.log('[Video Fullscreen Delay] 检测到退出全屏按钮点击');
            } else if (!isCurrentlyFullscreen) {
                //console.log('[Video Fullscreen Delay] 检测到进入全屏按钮点击');
                restartAudioCapture();
            } else {
                //console.log('[Video Fullscreen Delay] 检测到全屏按钮点击');
            }

            // 阻止默认行为
            event.preventDefault();
            event.stopPropagation();

            // 延迟执行点击
            setTimeout(() => {
                // 创建新的点击事件
                const newEvent = new MouseEvent('click', {
                    bubbles: true,
                    cancelable: true,
                    view: window
                });

                // 临时禁用监听器，避免无限循环
                isListenerActive = false;

                // 触发原始点击
                target.dispatchEvent(newEvent);

                // 重新启用监听器
                setTimeout(() => {
                    isListenerActive = true;
                }, 100);

            }, DELAY_TIME);
        }
    }

    document.addEventListener('click', clickHandler, true); // 使用捕获阶段
}

// 初始化函数
function initFullscreenDelay() {
    // 重写全屏API方法
    overrideFullscreenMethods();

    // 监听双击事件
    document.addEventListener('dblclick', handleDoubleClick, true);

    // 监听全屏按钮点击
    interceptFullscreenButtons();

    //console.log('[Video Fullscreen Delay] 插件已加载，全屏延迟功能已启用');
}

